module EncodingMaxPlus where

import Term
import Rule
import ReductionPair (ReductionPair (..))
import SMT
import qualified MaxPlus
import Order
import TRS

-- f_A(x1,...,xn) = max{a0, b1 (a1 + x1), ..., bn (an + xn)}

-- Encoding functions.

type MaxPlus = (Exp, [(Formula, Exp)])
type Interpretation = (String, MaxPlus)
type Algebra = [Interpretation]

-- (Nothing, b, a) denotes b a
-- (Just x, b, a)  denotes b (a + x)
type Affine = (Maybe String, Formula, Exp)

var_a0 :: Int -> String -> Exp
var_a0 k f = 
  Var ("a0_" ++ show k ++ "_" ++ f) 

var_ai :: Int -> String -> Int -> Exp
var_ai k f i =
  Var ("ai_" ++ show k ++ "_" ++ f ++ "_" ++ show i) 

var_bi :: Int -> String -> Int -> Formula
var_bi k f i =
  FVar ("bi_" ++ show k ++ "_" ++ f ++ "_" ++ show i) 

interpret :: Int -> Term -> [Affine]
interpret _ (V x)    = [(Just x, top, Val 0)]
interpret k (F f ts) =
  (Nothing, top, var_a0 k f) :
  [ (m, conj [var_bi k f i, b], plus [var_ai k f i, a]) 
  | (i, ti) <- zip [0..] ts,
    (m, b, a) <- interpret k ti]

geq_affine :: Affine -> Affine -> Formula
geq_affine (_, b1, a1) (Nothing, b2, a2) = 
  implies b2 (conj [b1, geq a1 a2])
geq_affine (Just x, b1, a1) (Just y, b2, a2)
  | x == y = implies b2 (conj [b1, geq a1 a2])
geq_affine _ _ = bottom

gt_affine :: Affine -> Affine -> Formula
gt_affine (_, b1, a1) (Nothing, b2, a2) = 
  implies b2 (conj [b1, gt a1 a2])
gt_affine (Just x, b1, a1) (Just y, b2, a2)
  | x == y = implies b2 (conj [b1, gt a1 a2])
gt_affine _ _ = bottom
-- Hoare extensions of geq_affine and gt_affine

geq_affine_hoare :: [Affine] -> [Affine] -> Formula
geq_affine_hoare as bs =
  conj [ disj [ geq_affine a b | a <- as ] | b <- bs ]

gt_affine_hoare :: [Affine] -> [Affine] -> Formula
gt_affine_hoare as bs =
  conj (disj [ b | (_, b, _) <- as ] :
        [ disj [ gt_affine a b | a <- as ] | b <- bs ])
 
-- s >=_A t
geq_A :: Int -> Rule -> Formula
geq_A k (s, t) = 
  geq_affine_hoare (interpret k s) (interpret k t)

-- s >_A t
gt_A :: Int -> Rule -> Formula
gt_A k (s, t) = 
  gt_affine_hoare (interpret k s) (interpret k t)

f_i_monotone :: Int -> (String, Int) -> Int -> Formula
f_i_monotone k (f, 1) 0 =
  conj [var_bi k f 0, 
        geq (var_ai k f 0) (var_a0 k f)]
f_i_monotone _ _ _ = bottom

f_i_invariant :: Int -> (String, Int) -> Int -> Formula
f_i_invariant k (f, _) i = neg (var_bi k f i)

strictly_monotone :: Int -> Signature -> Formula
strictly_monotone k sig =
  conj [ f_i_monotone k (f, n) i 
       | (f, n) <- sig, i <- [0..n-1] ]

weakly_simple :: Int -> (String, Int) -> Int -> Formula
weakly_simple k (f, _) i = conj [ var_bi k f i, geq (var_ai k f i) (Val 0) ]

strictly_simple :: Int -> (String, Int) -> Int -> Formula
strictly_simple k (f, _) i = conj [ var_bi k f i, gt (var_ai k f i) (Val 0) ]

side_condition :: Int -> (TRS, TRS) -> Signature -> Formula
side_condition k _ sig =
  conj [ geq (var_a0 k f) (Val 0) | (f, _) <- sig ]

-- decoding

decode_argument :: Model -> Int -> String -> Int -> Maybe Int
decode_argument model k f i
  | evalFormula model (var_bi k f i) = Just (evalExp model (var_ai k f i))
  | otherwise = Nothing 

decode_interpretation :: Model -> Int -> String -> Int -> MaxPlus.Interpretation
decode_interpretation model k f n =
  (f, (evalExp model (var_a0 k f),
       [ decode_argument model k f i | i <- [0..n-1] ]))

decode :: Int -> Model -> Signature -> Order
decode k model sig =
  MaxPlus [ decode_interpretation model k f n | (f, n) <- sig ]


reduction_pair :: Int -> ReductionPair.ReductionPair
reduction_pair k = ReductionPair.ReductionPair {
  _side_condition = side_condition k,
  _geq = geq_A k,
  _gt = gt_A k,
  _decode = decode k,
  _monotone = f_i_monotone k,
  _invariant = f_i_invariant k,
  _weakly_simple = weakly_simple k,
  _strictly_simple = strictly_simple k
}
