module Lex where

import Term
import Rule
import Order
import SMT
import ReductionPair (ReductionPair (..))
import TRS

geq_lex :: [ReductionPair] -> Rule -> Formula
geq_lex []         _    = top
geq_lex (rp : rps) rule =
  disj [_gt rp rule,
        conj [_geq rp rule, geq_lex rps rule]]

gt_lex :: [ReductionPair] -> Rule -> Formula
gt_lex []         _    = bottom
gt_lex (rp : rps) rule =
  disj [_gt rp rule,
        conj [_geq rp rule, gt_lex rps rule]]

monotone :: [ReductionPair] -> (String, Int) -> Int -> Formula
monotone rps (f, n) i =
  conj [_monotone rp (f, n) i | rp <- rps ]

invariant :: [ReductionPair] -> (String, Int) -> Int -> Formula
invariant rps (f, n) i =
  conj [_invariant rp (f, n) i | rp <- rps ]

f_i_combinable :: [ReductionPair] -> (String, Int) -> Int -> Formula
f_i_combinable [] _ _ = top
f_i_combinable (rp : rps) (f, n) i = 
  conj [disj [_monotone rp (f, n) i, invariant rps (f, n) i],
        f_i_combinable rps (f, n) i]

combinable :: [ReductionPair] -> Signature -> Formula
combinable rps sig =
  conj [ f_i_combinable rps (f, n) i
       | (f, n) <- sig,
         i <- [0..n-1] ]

side_condition :: [ReductionPair] -> (TRS, TRS) -> Signature -> Formula
side_condition rps p sig =
  conj (combinable rps sig :
        [_side_condition rp p sig | rp <- rps ])

decode :: [ReductionPair] -> Model -> Signature -> Order
decode rps model sig =
  Lex [ _decode rp model sig | rp <- rps ]

simple :: (String, Int) -> Int -> Formula
simple _ _ = error "simplicity of lex combination is not supported."

lex :: [ReductionPair] -> ReductionPair
lex rps = ReductionPair {
  _side_condition = side_condition rps,
  _geq = geq_lex rps,
  _gt = gt_lex rps,
  _monotone = monotone rps,
  _invariant = invariant rps,
  _decode = decode rps,
  _weakly_simple = simple,
  _strictly_simple = simple
}
