module Main where

import Data.List.Split (splitOn)
import Text.Read
import TRS
import DP
import Proof
import qualified Recursive as R
import qualified Matrix
import ReductionPair
import qualified Lex
import qualified EncodingMaxPlus
import qualified EncodingMatrix
import ARIParser
import qualified LPO
import Options.Applicative
import Options.Applicative.Help.Pretty
import qualified WPO

print_proof :: (TRS, TRS, Proof) -> IO ()
print_proof (rs, [], proof) = do
  putStrLn "We show the termination of the TRS R:"
  putStrLn ""
  putStr (showTRS rs)
  putStrLn ""
  putStr (show_proof proof)
print_proof (rs, ss, proof) = do
  putStrLn "We show the termination of the relative TRS R/S:"
  putStrLn ""
  putStrLn "  R:"
  putStr (showTRS rs)
  putStrLn ""
  putStrLn "  S:"
  putStr (showTRS ss)
  putStrLn ""
  putStr (show_proof proof)

read_maybe_number :: String -> Maybe Int
read_maybe_number s
  | Just n <- readMaybe s :: Maybe Int, n >= 0 = Just n
  | otherwise = Nothing

parse_base_order :: Int -> String -> ReductionPair
parse_base_order k ('S' : num)
  | Just d <- read_maybe_number num =
    EncodingMatrix.reduction_pair Matrix.Standard d k
parse_base_order k ('E' : num) 
  | Just d <- read_maybe_number num =
    EncodingMatrix.reduction_pair Matrix.Lex d k
parse_base_order k "M"  = EncodingMaxPlus.reduction_pair k
parse_base_order k "L"  = LPO.reduction_pair k
parse_base_order k ('W' : 'S' : num)
  | Just d <- read_maybe_number num =
    WPO.wpo k (EncodingMatrix.reduction_pair Matrix.Standard d k)
parse_base_order k ('W' : 'E' : num) 
  | Just d <- read_maybe_number num =
    WPO.wpo k (EncodingMatrix.reduction_pair Matrix.Lex d k)
parse_base_order k "WM" =
  WPO.wpo k (EncodingMaxPlus.reduction_pair k)
-- NOTE: more work is needed, for example, to support WL (WPO with LPO)
-- (neccesary avoid symbol collision in SMT encoding)
parse_base_order _ _    = error "parse_base_order"

parse_order :: String -> ReductionPair
parse_order s =
  case splitOn "+" s of
    []  -> error "Specify base orders."
    [name] -> parse_base_order 0 name
    names  -> Lex.lex [ parse_base_order i name 
                      | (i, name) <- zip [0..] names ]

parse_orders :: String -> [ReductionPair]
parse_orders s =
  case splitOn "," s of
    []    -> error "Specify base orders."
    names -> [ parse_order name | name <- names ]

prove_recursive :: Bool -> [R.Criterion] -> TRS -> TRS -> IO ()
prove_recursive oneshot criteria rs ss =
  case dp_problem rs ss of
    Nothing -> putStrLn "MAYBE"
    Just problem -> do
      result <- R.solve_problem oneshot criteria problem 
      case result of
        Nothing    -> do
          putStrLn "MAYBE"
        Just proof -> do
          putStrLn "YES"
          putStrLn ""
          print_proof (rs, ss, proof)

prove :: ReductionPair.Option -> [ReductionPair] -> TRS -> TRS -> IO ()
prove rp_option rps rs ss =
  prove_recursive 
    (_all_strict rp_option)
    [ process_with rp_option rp "z3" | rp <- rps] rs ss

dispatch :: Opt -> IO ()
dispatch opt = do
  errorOrTRS <- read_file (_file opt)
  case errorOrTRS of
    Left e    -> putStrLn (show e)
    Right (rs, ss)
      | TRS.variableCondition (rs ++ ss) -> 
        let rp_option = Option{
          _all_strict       = _oneshot opt,
          _use_rule_removal = not (_no_rule_removal opt),
          _use_usable_rules = not (_no_usable_rules opt)
          }
        in 
          prove rp_option (parse_orders (_orders opt)) rs ss
      | otherwise -> do
          putStrLn "NO"
          putStrLn ""
          putStrLn "The input TRS violates the variable condition and therefore is non-terminating."

-- command line options

data Opt = Opt { 
  _orders :: String,
  _no_rule_removal :: Bool,
  _no_usable_rules :: Bool,
  _oneshot :: Bool,
  _file :: String
}

option_spec :: Parser Opt
option_spec = Opt
  <$> strOption 
    (long "orders"
     <> short 'o'
     <> metavar "ORDERS"
     <> value "S1"
     <> help ("orderings: S<NUM>, E<NUM>, L, M, W{S<NUM>,E<NUM>,M} (e.g., -o E4+L,M,WS3)"))
  <*> switch
    (long "no-rule-removal"
     <> help "do not use the rule removal criterion")
  <*> switch
    (long "no-usable-rules"
     <> help "do not use the usable rule criterion")
  <*> switch
    (long "oneshot"
     <> help "strictly orient all rules in DP(R)")
  <*> strArgument
    (metavar "TARGET"
     <> help "target file in ARI/WST format")

order_descriptions :: Doc
order_descriptions =
  vsep 
    [pretty "Orderings:",
     pretty "  S<d=NUM>            d-dimensional standard matrix interpretation", 
     pretty "  E<d=NUM>            d-dimensional echelon-form matrix interpretation",
     pretty "  M                   max/plus interpretation",
     pretty "  L                   LPO",
     pretty "  W{S<NUM>,E<NUM>,M}  WPO with matrix or max/plus interpretation"]
-- TODO: learn optparse or replace it

opts :: ParserInfo Opt
opts =
  info (helper <*> option_spec)
       (fullDesc <> 
        header "Mint -- Mini Termination Tool verion 0.3" <>
        footerDoc (Just order_descriptions))

main :: IO ()
main = do
  result <- execParser opts
  dispatch result
