-- Matrix interpretations equipped with
--  - the standard order on vectors
--  - the lexicographic order on vectors 
module Matrix where

import Data.List

type Vector = [Int]
type Matrix = [Vector]
type Linear = (Int, [Int])
type Form = (Vector, [Matrix])
type Interpretation = (String, Form)
data BaseOrder = Standard | Lex
type Algebra = (BaseOrder, Int, [Interpretation])

is_zero_vector :: Vector -> Bool
is_zero_vector v = all (== 0) v

is_zero_matrix :: Matrix -> Bool
is_zero_matrix m = all is_zero_vector m

is_unit_matrix :: Matrix -> Bool
is_unit_matrix m =
  and [ (i == j && m_ij == 1) || (i /= j && m_ij == 0)
      | i <- [0..d-1],
        j <- [0..d-1],
        let m_ij = m !! i !! j ]
  where d = length m

show_vector :: Vector -> String
show_vector [a] = show a
show_vector v   = 
  "(" ++ intercalate "," [ show a | a <- v ] ++ ")"

show_matrix :: Matrix -> String
show_matrix [[a]] = show a
show_matrix m =
  "(" ++ intercalate "," [ show_vector v | v <- m ] ++ ")" 

show_m_xi :: (Matrix, Int) -> String
show_m_xi (m, i)
  | is_unit_matrix m = "x" ++ show i
  | otherwise        = show_matrix m ++ " x" ++ show i

show_form :: Form -> String
show_form (v, ms)
  | ss == [] = s
  | is_zero_vector v = intercalate " + " ss
  | otherwise        = intercalate " + " (ss ++ [s])
  where
    s  = show_vector v
    ss = [ show_m_xi (m, i) 
         | (m, i) <- zip ms [1 :: Int ..],
           not (is_zero_matrix m) ]

show_lhs :: (String, Int) -> String
show_lhs (f, n) =
  f ++ "_A(" ++ intercalate "," [ "x" ++ show i | i <- [1..n] ] ++ ")"

show_interpretation :: Interpretation -> String
show_interpretation (f, form@(_, ms)) =
  show_lhs (f, length ms) ++ " = " ++ show_form form

show_order :: BaseOrder -> String
show_order Standard = "standard order"
show_order Lex = "lexicographic order"

show_algebra :: Algebra -> String
show_algebra (order, d, interpretations) =
  unlines 
    (["  carrier: N^" ++ show d,
      "  order: " ++ show_order order,
      "  interpretations:" ] ++
     [ "    " ++ show_interpretation f_A | f_A <- interpretations ])
