module MaxPlus where

import Data.List
import Term
import Rule


-- f_A(x1,...,xn) = max{a0, (a_i1 + x_i1), ..., (a_im + x_im)}
-- for some 1 <= i1 < ... < im <= n

type MaxPlus = (Int, [Maybe Int])
type Interpretation = (String, MaxPlus)
type Algebra = [Interpretation]
type Affine = (Int, Maybe String)

show_monomial :: (Int, String) -> String
show_monomial (0, _) = "0"
show_monomial (1, x) = x
show_monomial (n, x) = show n ++ " " ++ x

-- pretty printer for xi + a
show_xi_plus_a :: (Int, Int) -> String
show_xi_plus_a (i, 0) = "x" ++ show i
show_xi_plus_a (i, a)
  | a > 0     = "x" ++ show i ++ " + " ++ show a
  | otherwise = "x" ++ show i ++ " - " ++ show (- a) 

show_max_plus :: MaxPlus -> String
show_max_plus (a0, as) =
  case ss of
    []  -> show a0
    [s] -> s
    _   -> "max{" ++ intercalate ", " ss ++ "}"
  where 
    ss' = [ show_xi_plus_a (i, a) | (i, Just a) <- zip [1..] as ]
    ss = if any (>= a0) [ ai | Just ai <- as ] then ss' else show a0 : ss'

show_lhs :: (String, Int) -> String
show_lhs (f, 0) = f ++ "_A"
show_lhs (f, n) = 
  f ++ "_A(" ++ 
  intercalate "," [ "x" ++ show i | i <- [1::Int ..n] ] ++ 
  ")"

show_interpretation :: Interpretation -> String
show_interpretation (f, (a0, as)) =
  show_lhs (f, length as) ++ " = " ++
  show_max_plus (a0, as)

show_algebra :: Algebra -> String
show_algebra a =
  unlines [ show_interpretation f_A | f_A <- a ]

interpret :: Algebra -> Term -> [Affine] 
interpret _ (V x) = [(0, Just x)]
interpret a (F f ts)
  | Just (a0, as) <- lookup f a =
    (a0, Nothing) :
    [ (ai + n, m) 
    | (Just ai, ti) <- zip as ts,
      (n, m) <- interpret a ti]
  | otherwise = error "interpret"

geq_affine :: Affine -> Affine -> Bool
geq_affine (a, _)      (b, Nothing) = a >= b
geq_affine (a, Just x) (b, Just y)  = a >= b && x == y
geq_affine _           _            = False

gt_affine :: Affine -> Affine -> Bool
gt_affine (a, _)      (b, Nothing) = a > b
gt_affine (a, Just x) (b, Just y)  = a > b && x == y
gt_affine _           _            = False

-- Hoare extensions

geq_affine_hoare :: [Affine] -> [Affine] -> Bool
geq_affine_hoare as bs =
  all (\b -> any (\a -> geq_affine a b) as) bs

gt_affine_hoare :: [Affine] -> [Affine] -> Bool
gt_affine_hoare as bs =
  as /= [] && all (\b -> any (\a -> gt_affine a b) as) bs

geq_A :: Algebra -> Rule -> Bool
geq_A a (s, t) =
  geq_affine_hoare (interpret a s) (interpret a t)

gt_A :: Algebra -> Rule -> Bool
gt_A a (s, t) =
  gt_affine_hoare (interpret a s) (interpret a t)

f_i_monotone :: Algebra -> (String, Int) -> Int -> Bool
f_i_monotone a (f, 1) 1
  | Just (a0, [Just a1]) <- lookup f a = a1 >= a0
f_i_monotone _ _ _ = False

f_i_invariant :: Algebra -> (String, Int) -> Int -> Bool
f_i_invariant a (f, _) i
  | Just (_, as) <- lookup f a = as !! i == Nothing
f_i_invariant _ _ _ = False

strictly_monotone_interpretation :: Interpretation -> Bool
strictly_monotone_interpretation (_, (_, []))         = True
strictly_monotone_interpretation (_, (a0, [Just a1])) = a1 >= a0
strictly_monotone_interpretation _                    = False

strictly_monotone :: Algebra -> Bool
strictly_monotone a =
  all strictly_monotone_interpretation a
