module Order where

import qualified Linear
import qualified MaxPlus 
import Matrix
import Precedence
import ArgumentFiltering

data Order =
    Linear Linear.Algebra
  | Matrix Matrix.Algebra
  | MaxPlus MaxPlus.Algebra
  | LPO Precedence ArgumentFiltering
  | Lex [Order]
  | WPO Order Precedence ArgumentFiltering

indent :: Int -> String -> String
indent n text =
  unlines [ replicate n ' ' ++ line | line <- lines text ] 

instance Show Order where
  show (Linear a) = 
    "linear polynomial interpretations on natural numbers:\n\n" ++
    indent 2 (Linear.show_algebra a)
  show (Matrix a) =
    "matrix interpretations:\n\n" ++
    Matrix.show_algebra a
  show (MaxPlus a) =
    "max/plus interpretations on natural numbers:\n\n" ++
    indent 2 (MaxPlus.show_algebra a)
  show (LPO p af) =
    "lexicographic path order\n\n" ++
    indent 2 (showPrecedence p) ++
    "\n" ++
    "  argument filter:\n\n" ++
    indent 2 (showArgumentFiltering af)
  show (Order.Lex os) =
    "lexicographic combination of reduction pairs:\n\n" ++
    indent 2 (concat [ show i ++ ". " ++ show o ++ "\n" 
                     | (i, o) <- zip [1::Int ..] os ])
  show (WPO o p af) =
    "weighted path order\n\n" ++
    "  base order:\n\n" ++
    indent 4 (show o) ++
    "\n" ++
    indent 2 (showPrecedence p) ++
    "  partial status:\n\n" ++
    indent 4 (showArgumentFiltering af)
