module Parser (readTRSFile) where

import Data.List
import TRS
import Term
import Rule
import Text.ParserCombinators.Parsec

variables :: Term -> [String]
variables (V x)    = [x]
variables (F _ ts) = nub [ x | t <- ts, x <- Parser.variables t ]

variablesInTRS :: TRS -> [String]
variablesInTRS trs =
  nub [ x | (l, r) <- trs, t <- [l, r], x <- Parser.variables t ]

substitute :: Term -> Subst -> Term
substitute (V x) sigma
    | Just t <- lookup x sigma = t
    | otherwise                = V x
substitute (F f ts) sigma      = F f [ Parser.substitute t sigma | t <- ts ]

substituteTRS :: TRS -> Subst -> TRS
substituteTRS trs sigma =
    [ (Parser.substitute l sigma, Parser.substitute r sigma) | (l, r) <- trs ]

convert :: [String] -> TRS -> TRS
convert xs trs = substituteTRS trs sigma
    where sigma = [ (x, F x []) | x <- Parser.variablesInTRS trs, not (elem x xs) ]

-- Scanners.

identifier :: Parser String
identifier = do
  spaces
  x <- many1 (noneOf "(), \t\r\n")
  spaces
  return x

keyword :: String -> Parser ()
keyword s = do
  spaces
  _ <- string s
  spaces
  return ()

-- Parsing functions.
            
parseTerm :: Parser Term
parseTerm = try parseFunction <|> parseVariable

parseVariable :: Parser Term
parseVariable = do
  x <- identifier
  return (V x)

parseFunction :: Parser Term
parseFunction = do
  f <- identifier
  keyword "("
  t <- sepBy parseTerm (keyword ",")
  keyword ")"
  return (F f t)

data Rule' = Strict Rule | Relative Rule

parseRule :: Parser Rule'
parseRule = do
  l <- parseTerm
  keyword "->"
  r <- parseTerm
  return (Strict (l, r))

parseRelativeRule :: Parser Rule'
parseRelativeRule = do
  l <- parseTerm
  keyword "->="
  r <- parseTerm
  return (Relative (l, r))

-- wrapper
parseRule' :: Parser Rule'
parseRule' = try parseRelativeRule <|> parseRule

parseVAR :: Parser ([String], TRS, TRS)
parseVAR = do
  keyword "VAR"
  xs <- many identifier
  return (xs, [], [])

parseRULES :: Parser ([String], TRS, TRS)
parseRULES = do
  keyword "RULES"
  rs <- many parseRule'
  return ([], collectS rs, collectR rs)
  where
    collectS [] = []
    collectS (Strict rule : rs) = rule : collectS rs
    collectS (Relative _ : rs) = collectS rs
    collectR [] = []
    collectR (Strict _ : rs) = collectR rs
    collectR (Relative rule : rs) = rule : collectR rs

parseAnything :: Parser ()
parseAnything =
  do { _ <- identifier; return () } <|>
  do { keyword "("; _ <- many parseAnything; keyword ")" }

parseComment :: Parser ([String], TRS, TRS)
parseComment = do
  _ <- many parseAnything
  return ([], [], [])

parseSection :: Parser ([String], TRS, TRS)
parseSection = do
  keyword "("
  (xs, trs1, trs2) <- try parseVAR <|> try parseRULES <|> parseComment
  keyword ")"
  return (xs, trs1, trs2)

parseTRS :: Parser (TRS, TRS)
parseTRS = do
  ps <- many parseSection
  eof
  let (xss, trss1, trss2) = unzip3 ps in
    return (convert (concat xss) (concat trss1), convert (concat xss) (concat trss2))

readTRSFile :: String -> IO (Either ParseError (TRS, TRS))
readTRSFile path = parseFromFile parseTRS path
