module Proof where

import Data.List
import Rule
import TRS
import DP
import Order

data CriterionProof = 
    SCCs { 
      _problem :: Problem,
      _subproblems :: [Problem]
    }
  | ReductionPair {
      _problem :: Problem,
      _order :: Order,
      _monotone_reduction_pair :: Bool,
      _usable_rules :: TRS,
      _subproblem :: Problem
    }

type Subproof = (CriterionProof, [Problem])

data Proof = Node Problem CriterionProof [Proof]

show_indexed_rules :: (String, TRS) -> String
show_indexed_rules (_, []) =
  "  (no rules)\n"
show_indexed_rules (s, trs) =
  unlines [ s ++ show i ++ ": " ++ showRule rule 
          | (i, rule) <- zip [1::Int ..] trs ]

show_indexes :: (String, TRS, TRS) -> String
show_indexes (_, _, []) = "(no rules)"
show_indexes (s, trs, subsystem) =
  intercalate ", "
    [ s ++ show i 
    | (i, rule) <- zip [1 :: Int ..] trs, elem rule subsystem ]

caption :: CriterionProof -> String
caption SCCs{}          = "-- SCC decomposition."
caption ReductionPair{} = "-- Reduction pair."

show_SCC :: (TRS, TRS) -> String
show_SCC (ps, scc) =
  "  {" ++
  intercalate ", " 
    [ "p" ++ show i 
    | (i, rule) <- zip [1 :: Int ..] ps, elem rule scc ] ++
  "}"

show_SCCs :: (Problem, [Problem]) -> String
show_SCCs (_, []) = "  (no SCCs)\n"
show_SCCs ((ps, _, _), problems) =
  unlines [ show_SCC (ps, scc) | (scc, _, _) <- problems ] ++ "\n"

instance Show CriterionProof where
  show proof@SCCs{} =
    "The estimated dependency graph contains the following SCCs:\n\n" ++ 
    show_SCCs (_problem proof, _subproblems proof)
  show proof@ReductionPair{} =
    "The set of usable rules consists of\n\n" ++
    "  " ++ show_indexes ("r", rs, _usable_rules proof) ++ "\n\n" ++
    "Take the " ++ 
    (if _monotone_reduction_pair proof then "monotone " else "") ++
    "reduction pair:\n\n" ++
    indent 2 (show (_order proof)) ++ 
    "\n" ++
    "The next rules are strictly ordered:\n\n" ++
    (if removed_from_P == [] then 
       ""
     else
      "  " ++ show_indexes ("p", ps, removed_from_P) ++ "\n") ++
    (if removed_from_R == [] then
       ""
     else
       "  " ++ show_indexes ("r", rs, removed_from_R) ++ "\n") ++
    "\n" ++
    "We remove them from the problem." ++
    (if ps' == [] then "  Then no dependency pair remains.\n" else "\n")
    where
      (ps,  rs,  _) = _problem proof
      (ps', rs', _) = _subproblem proof
      removed_from_P = TRS.diff ps ps'
      removed_from_R = TRS.diff rs rs'

show_proof :: Proof -> String
show_proof (Node (ps, rs, m) criterion_proof subproofs) =
  caption criterion_proof ++ "\n\n" ++
  "Consider the " ++
  (if m then "" else "non-minimal ") ++
  "dependency pair problem (P, R), where P consists of\n\n" ++
  show_indexed_rules ("p", ps) ++ "\n" ++
  "and R consists of:\n\n" ++
  show_indexed_rules ("r", rs) ++ "\n" ++
  show criterion_proof ++ 
  "\n" ++ 
  concat [ show_proof subproof | subproof <- subproofs ]
