module ReductionPair where

import Term
import Rule
import TRS
import DP
import Order
import SMT
import Usable
import Proof

data ReductionPair = ReductionPair {
  _side_condition :: (TRS, TRS) -> Signature -> Formula, -- first TRS to be strictly oriented, second weakly
  _geq :: Rule -> Formula,
  _gt :: Rule -> Formula,
  _monotone :: (String, Int) -> Int -> Formula, -- strict monotonicity, 0-indexed
  _invariant :: (String, Int) -> Int -> Formula, -- 0-indexed
  _weakly_simple :: (String, Int) -> Int -> Formula, -- weak simplicity, for WPO, 0-indexed
  _strictly_simple :: (String, Int) -> Int -> Formula, -- strict simplicity, for WPO, 0-indexed
  _decode :: Model -> Signature -> Order
}

data Option = Option {
  _all_strict :: Bool, -- P subseteq >
  _use_rule_removal :: Bool,
  _use_usable_rules :: Bool
}

var_gt :: String -> Int -> Formula
var_gt s i = FVar ("gt_" ++ s ++ "_" ++ show i)

monotonicity :: ReductionPair -> Signature -> Formula
monotonicity rp sig =
  conj [ _monotone rp (f, n) i
       | (f, n) <- sig,
         i <- [0..n-1] ]

encode :: Option -> ReductionPair -> Problem -> (TRS, Formula)
encode option rp (ps, rs, minimal) =
  (us, 
   conj (_side_condition rp (ps, us) sig :
         [ iff (var_gt "P" i) (_gt rp rule) | (i, rule) <- ips ] ++
         [ iff (var_gt "R" i) (_gt rp rule) | (i, rule) <- ius ] ++
         (if _all_strict option then
            [ _geq rp rule | rule <- us ] ++
            [ var_gt "P" i | i <- [0 .. length ps - 1] ]
          else
            [ _geq rp rule | rule <- ps ++ us ] ++
            [disj (rule_removal_constraint :
                   [ var_gt "P" i | i <- [0 .. length ps - 1] ])]
         )
        )
  )
  where
    us = if _use_usable_rules option && minimal then 
           usable_rules ps rs
         else 
           rs
    sig = TRS.signatureOf (ps ++ us)
    ips = zip [0..] ps
    ius = zip [0..] us
    rule_removal_constraint
      | _use_rule_removal option =
         conj [monotonicity rp sig, disj [ var_gt "R" i | (i, _) <- ius ]]
      | otherwise = bottom

monotone :: Model -> ReductionPair -> Signature -> Bool
monotone model rp sig = 
  and [ evalFormula model (_monotone rp (f, n) i)
      | (f, n) <- sig,
        i <- [0..n-1] ]

reduced_problem :: Option -> ReductionPair -> Model -> Problem -> TRS -> Problem
reduced_problem param rp model (ps, rs, minimal) us =
  ([ rule 
   | (i, rule) <- zip [0..] ps,
     not (evalFormula model (var_gt "P" i)) ],
   if _use_rule_removal param && monotone model rp sig then
     [ rule 
     | (i, rule) <- zip [0..] us,
       not (evalFormula model (var_gt "R" i)) ]
   else
     rs,
   minimal)
  where
    sig = TRS.signatureOf (ps ++ us)

process_with :: Option -> ReductionPair -> String -> Problem -> IO (Maybe Subproof)
process_with param rp tool problem@(ps, _, _) = do
  result <- sat tool formula
  case result of
    Nothing    -> return Nothing
    Just model ->
      let
        subproblem = reduced_problem param rp model problem us
        proof = Proof.ReductionPair{
          _problem = problem,
          _monotone_reduction_pair = monotone model rp sig,
          _order = _decode rp model sig,
          _usable_rules = us,
          _subproblem = subproblem
        }
      in
        case subproblem of
          ([],_,_) -> return (Just (proof, []))
          _ -> return (Just (proof, [subproblem]))
  where
    (us, formula) = encode param rp problem
    sig = TRS.signatureOf (ps ++ us)

process :: ReductionPair -> String -> Problem -> IO (Maybe Subproof)
process = 
  process_with Option{
    _all_strict = False,
    _use_rule_removal = True,
    _use_usable_rules = True
  }
