module Term where

import Data.List

type Signature = [(String, Int)]

type ReplacementMap = [(String, [Bool])]

data Term = V String | F String [Term] deriving Eq

type Subst = [(String, Term)]

type Position = [Int]

instance Show Term where
  show (V x)    = x
  show (F f ts) = f ++ "(" ++ intercalate "," [show t | t <- ts] ++ ")"

-- Var(t)

variables :: Term -> [String]
variables (V x) = [x]
variables (F _ ts) = nub [ x | t <- ts, x <- variables t ]

isVariable :: Term -> Bool
isVariable (V _) = True
isVariable _ = False

-- check if a term is of the form f(x_1, ..., x_n)
shallow :: Term -> Bool
shallow (V _) = False
shallow (F _ ts) = all isVariable ts

-- Fun(t)

functions :: Term -> [String]
functions (V _)    = []
functions (F f ts) = nub (f : [ g | t <- ts, g <- functions t ])

-- |t|
size :: Term -> Int
size (V _)    = 1
size (F _ ts) = 1 + sum [ size t | t <- ts ]

-- |t|_x
count :: Term -> String -> Int
count (V y) x
  | x == y    = 1
  | otherwise = 0
count (F _ ts) x = sum [ count t x | t <- ts ]

-- The list of all subterms of a term.
subterms :: Term -> [Term]
subterms t@(V _) = [t]
subterms t@(F _ us) = t : nub [ v | u <- us, v <- subterms u ]

proper_subterms :: Term -> [Term]
proper_subterms (V _)    = []
proper_subterms (F _ ts) = [ u | t <- ts, u <- subterms t ]

-- Pos(t)
positions :: Term -> [Position]
positions (V _)    = [ [] ]
positions (F _ ts) = [] : [ i : p | (i, t) <- zip [0..] ts, p <- positions t ]

-- Pos_F(t)
functionPositions :: Term -> [Position]
functionPositions (V _)    = []
functionPositions (F _ ts) = 
  [] : [ i : p | (i, t) <- zip [0..] ts, p <- functionPositions t ] 

-- t|_p
subtermAt :: Term -> Position -> Term
subtermAt t        []      = t
subtermAt (F _ ts) (i : p) = subtermAt (ts !! i) p
subtermAt _        _       = error "subtermAt"

-- t[u]_p
replace :: Term -> Term -> Position -> Term
replace _        u []      = u
replace (F f ts) u (i : p) =
  F f [ if i == j then replace tj u p else tj | (j, tj) <- zip [0..] ts ]
replace _ _ _ = error "replace"

signatureOf :: Term -> Signature
signatureOf t = nub [ (f, length ts) | F f ts <- subterms t ]

-- t^#
-- TODO: should be named ``sharp''?
mark :: Term -> Term
mark (V _)    = error "mark"
mark (F f ts) = F (f ++ "#") ts

bang :: Term -> Term
bang (V _) = error "cannot bang variables"
bang (F f ts) = F (f ++ "!") ts

-- rename "x" (f(x,g(y),x)) = f(x1,g(x2),x1)
rename :: String -> Term -> Term
rename x t = substitute t sigma
  where 
    sigma = zip (variables t) [ V (x ++ show i)| i <-  [1 :: Int ..] ]
  
-- t sigma
substitute :: Term -> Subst -> Term
substitute (V x) sigma
    | Just t <- lookup x sigma = t
    | otherwise                = V x
substitute (F f ts) sigma      = F f [ substitute t sigma | t <- ts ]

-- Dom(sigma)
domain :: Subst -> [String]
domain sigma = nub [ x | (x, _) <- sigma ]


-- x sigma tau = (x sigma) tau for all variables x
compose :: Subst -> Subst -> Subst
compose sigma tau =
  [ (x, substitute (substitute (V x) sigma) tau)
  | x <- nub (domain sigma ++ domain tau) ]


substituteES :: [(Term, Term)] -> Subst -> [(Term, Term)]
substituteES trs sigma = 
  [ (substitute l sigma, substitute r sigma) | (l, r) <- trs ]

-- Most general unifier

mgu' :: Subst -> [(Term, Term)] -> Maybe Subst
mgu' sigma [] = Just sigma
mgu' sigma ((V x, V y) : es)
  | x == y = mgu' sigma es
mgu' sigma ((V x, t) : es)
  | notElem x (variables t) =
      mgu' (compose sigma tau) (substituteES es tau)
  where tau = [(x, t)]
mgu' sigma ((s, t@(V _)) : es) = mgu' sigma ((t, s) : es)
mgu' sigma ((F f ss, F g ts) : es)
  | f == g = mgu' sigma (zip ss ts ++ es)
mgu' _ _ = Nothing

mgu :: Term -> Term -> Maybe Subst
mgu s t = mgu' [] [(s, t)]

unifiable :: Term -> Term -> Bool
unifiable s t = mgu s t /= Nothing
