module Rewriting where

import Data.List
import Term
import Rule
import TRS

-- { t | s ->_R^i t for some i <= k }
at_most_k_steps :: Int -> TRS -> Term -> [Term]
at_most_k_steps 0 _ s = [ s ]
at_most_k_steps k trs s =
  nub (s : [ u | t <- reducts trs s, u <- at_most_k_steps (k-1) trs t ])

at_most_k_walks :: Int -> TRS -> Term -> [(Term, [(Term, Term)])]
at_most_k_walks 0 _ s = [ (s, []) ]
at_most_k_walks k trs s =
  nub (ts ++ [ (u, nub (lr:rs)) | lr <- trs, (t,rs) <- ts, u <- reducts [lr] t ])
    where
      ts = at_most_k_walks (k-1) trs s

-- { t | s ->_R^i t for some 1 <= i <= k }
at_least_1_at_most_k_steps :: Int -> TRS -> Term -> [Term]
at_least_1_at_most_k_steps 0 _   _ = []
at_least_1_at_most_k_steps k trs t =
  [ v | u <- reducts trs t, v <- at_most_k_steps (k - 1) trs u ]

-- { t | s ->_R^epsilon t }
rootStep :: TRS -> Term -> [Term]
rootStep trs s =
  [ Term.substitute r sigma | (l, r) <- trs, Just sigma <- [match l s] ]

-- { t | s -||->_R t }
parallelStep :: TRS -> Term -> [Term]
parallelStep _ s@(V _) = [s]
parallelStep trs s@(F f ss) =
  nub (s : rootStep trs s ++
       [ F f ts | ts <- sequence [ parallelStep trs si | si <- ss ] ])

{-
-- variable-restricted parallel step
-- { t | s -||->_R^P t and Var(t, P) subseteq X }
restrictedParallelStep :: TRS -> [String] -> Term -> [Term]
restrictedParallelStep _ _ s@(V _)    = [s]
restrictedParallelStep trs xs s@(F f ss) =
  nub ([ t | t <- rootStep trs s, subset (Term.variables t) xs ] ++
       [ F f ts | ts <- sequence [ restrictedParallelStep trs xs si | si <- ss ] ])
-}


-- { (rule, t) | s ->_rule^epsilon t }
rootStepWithRule :: TRS -> Term -> [(Rule, Term)]
rootStepWithRule trs s =
  [ ((l, r), Term.substitute r sigma) 
  | (l, r) <- trs, 
    Just sigma <- [match l s] ]

-- variable-restricted parallel step
-- { (S, t) | S subseteq R, s -||->_S^P t and Var(t, P) subseteq X }
restrictedParallelStep :: TRS -> [String] -> Term -> [(TRS, Term)]
restrictedParallelStep _ _ s@(V _)    = [([], s)]
restrictedParallelStep trs xs s@(F f ss) =
  nub ([ ([rule], t) 
       | (rule, t) <- rootStepWithRule trs s, 
         subset (Term.variables t) xs ] ++
       [ (nub (concat ruless), F f ts) 
       | pairs <- sequence [ restrictedParallelStep trs xs si | si <- ss ],
         let (ruless, ts) = unzip pairs ])

-- { t | s -o->_R t }
multistep :: TRS -> Term -> [Term]
multistep _   s@(V _)    = [s]
multistep trs s@(F f ss) =
  nub (s : 
       [ Term.substitute r tau 
       | (l, r) <- trs, 
         Just sigma <- [match l s],
        tau <- multistepSubst trs sigma ] ++
        [ F f ts | ts <- sequence [ multistep trs si | si <- ss ] ])

multistepSubst :: TRS -> Subst -> [Subst]
multistepSubst trs sigma =
  sequence [ [ (x, t) | t <- multistep trs s ] | (x, s) <- sigma ] 
