module TRS where
    
import Data.List
import Term
import Rule

type TRS       = [Rule]
type Overlap   = (Rule, Position, Rule, Subst)

type Equation = (Term, Term)
type ES = [Equation]

showTRS :: TRS -> String
showTRS trs = unlines [ "  " ++ showRule rule | rule <- trs ]

showEq :: Rule -> String
showEq (l, r) = show l ++ " = " ++ show r

showES :: TRS -> String
showES es = unlines [ "  " ++ showEq e | e <- es ]

signatureOf :: TRS -> [(String, Int)]
signatureOf trs = nub [ fn | rule <- trs, fn <- Rule.signatureOf rule ]

substitute :: TRS -> Subst -> TRS
substitute trs sigma = [ Rule.substitute rule sigma | rule <- trs ]

variableCondition :: TRS -> Bool
variableCondition trs = all Rule.variableCondition trs

variables :: TRS -> [String]
variables trs = nub [ x | rule <- trs, x <- Rule.variables rule ]

-- Fun(R)

functions :: TRS -> [String]
functions trs = nub [ f | rule <- trs, f <- Rule.functions rule ]

functionsES :: ES -> [String]
functionsES es = TRS.functions es

-- D_R

definedSymbols :: TRS -> [String]
definedSymbols trs = nub [ f | (F f _, _) <- trs ]

substituteES :: TRS -> Subst -> TRS
substituteES = TRS.substitute

lhss :: TRS -> [Term]
lhss trs = [ l | (l, _) <- trs ]

linear :: TRS -> Bool
linear trs = all Rule.linear trs

leftLinear :: TRS -> Bool
leftLinear trs = all Rule.leftLinear trs

variantTRS :: TRS -> TRS -> Bool
variantTRS trs1 trs2 =
  all (\lr -> any (Rule.variant lr) trs2) trs1
  &&
  all (\lr -> any (Rule.variant lr) trs1) trs2

subsume :: TRS -> TRS -> Bool
subsume trs1 trs2 =
  all
    (\lr2 -> any (\lr1 -> Rule.subsume lr1 lr2) trs1)
    trs2

unique :: TRS -> TRS
unique trs = nubBy Rule.variant trs

-- Rewriting

normalForm :: TRS -> Term -> Bool
normalForm trs t = rewrite trs t == Nothing

reducible :: TRS -> Term -> Bool
reducible trs t = rewrite trs t /= Nothing

-- [ u | t ->_R u ]
reducts :: TRS -> Term -> [Term]
reducts trs t =
  [ replace t (Term.substitute r sigma) p
  | p <- positions t,
    (l, r) <- trs,
    Just sigma <- [match l (subtermAt t p)] ]


-- A term u if t ->_R u.
rewrite :: TRS -> Term -> Maybe Term
rewrite trs t =
  case sortOn Term.size (reducts trs t) of
    []    -> Nothing
    u : _ -> Just u

rename :: String -> Int -> TRS -> TRS
rename x k trs = [ Rule.rename x k rule | rule <- trs ]

-- assume in-out critical pair: R<- x| -e>S
overlap2 :: TRS -> TRS -> [Overlap]
overlap2 rr ss =
  [ ((l1, r1), p, (l2, r2), sigma)
  | rule2 <- ss,
    let (l2, r2) = Rule.rename "y" 0 rule2,
    p <- functionPositions l2,
    rule1 <- rr,
    p /= [] || not (Rule.variant rule1 rule2),
    let (l1, r1) = Rule.rename "x" 0 rule1,
    Just sigma <- [mgu l1 (subtermAt l2 p)] ]

variantOverlap :: (Term,Term,Term) -> (Term,Term,Term) -> Bool
variantOverlap (a,b,c) (d,e,f) =
  Term.variant (F "_" [a,b,c]) (F "_" [d,e,f])

-- critical pairs: R<- x| ->S
cpeak2 :: TRS -> TRS -> [(Term,Term,Term)]
cpeak2 rr ss =
  nubBy variantOverlap
      [ (ren t, ren s, ren u)
      | ((_, r1), p, (l2, r2), sigma) <- overlap2 rr ss,
        let s = Term.substitute l2 sigma,
        let t = Term.substitute (replace l2 r1 p) sigma,
        let u = Term.substitute r2 sigma ]
  where
    ren t = Term.substitute t tau
    ys = TRS.variables (rr ++ ss)
    tau = [(y, V ("x" ++ show i)) | (y, i) <- zip ys [0 :: Int ..]]

allCpeak2 :: TRS -> TRS -> [(Term,Term,Term)]
allCpeak2 rr ss = nubBy variantOverlap (xs ++ ys)
  where
    xs = cpeak2 rr ss
    ys = [(t,s,u) | (u,s,t) <- cpeak2 ss rr]

-- critical pairs: R<- x| ->S
cp2 :: TRS -> TRS -> TRS
cp2 rr ss = nubBy Rule.variant [(t,u) | (t, _, u) <- cpeak2 rr ss]

overlap :: TRS -> [Overlap]
overlap trs = overlap2 trs trs

-- critical pairs
cp :: TRS -> TRS
cp trs = cp2 trs trs

