-- Critical Pair Closing

module CPS (sh22, cpsWithC, hm11) where

import Data.List
import Term
import Rule (subset)
import TRS
import Result
import Rewriting
import SN
import CompositionalCriteria
import qualified Proof as Proof

proof :: TRS -> TRS -> TRS -> String -> String
proof trs cs ps crProof =
  Proof.subproof
    "Compositional critical pair system (Shintani and Hirokawa 2022)."
    trs cs
    (unlines
      [ "The critical pair system CPS(R,C) is:"
      , ""
      , Proof.showTRS ps
      , "All pairs in PCP(R) are joinable and PCPS(R,C)/R is terminating." ])
    crProof

-- Shintani and Hirokawa 2022
-- Search a suitable subsystem.
sh22 :: RTermination -> Int-> Criterion -> TRS -> IO Result
sh22 tt k cr trs
  | TRS.leftLinear trs && wcr k trs = do
      cpsCriterion tt cr k trs (initializeCs trs)
  | otherwise = return MAYBE

-- return YES if there is a subsystem cs in css s.t.
--   CR(cs) and CPS(trs,cs)/trs is SN
cpsCriterion :: RTermination -> Criterion -> Int -> TRS -> [TRS] -> IO Result
cpsCriterion _ _ _ _ [] = return MAYBE
cpsCriterion tt cr k trs (cs:css) = do
  m <- tt ps trs
  case m of
    YES _ -> do
      v <- cr cs
      case v of
        YES crProof ->
          return (YES (proof trs cs ps crProof))
        NO _ | subset trs cs ->
          return v
        _ ->
          cpsCriterion tt cr k trs css
    _ -> cpsCriterion tt cr k trs css
  where
    ps = cps trs cs

-- Shintani and Hirokawa 2022
-- ** Assume that C is confluent. **
cpsWithC :: RTermination -> Int-> TRS -> Criterion
cpsWithC tt k trs cs
  | TRS.leftLinear trs && wcr k trs = do
    m <- tt ps trs
    case m of
      YES _ -> return (YES (proof trs cs ps "  (assumption)"))
      _     -> return MAYBE
  | otherwise = return MAYBE
  where
    ps = cps trs cs

initializeCs :: TRS -> [TRS]
initializeCs trs =
  trs : [c | c1 <- power (trs \\ c0),
             let c = c1 ++ c0,
             -- c =/= trs
             length c /= n]
  where
    c0 = makeC0 trs
    n  = length trs

makeC0 :: TRS -> TRS
makeC0 trs =
  nub $ concat
    [[a,b] | a <- trs, b <- trs,
             (t,_,s,u) <- TRS.cpeak2 [b] [a],
             looping [(s,t),(s,u)] trs]

-- Hirokawa and Middeldorp 2011
hm11 :: RTermination -> Int -> Criterion
hm11 tt k trs
  | TRS.leftLinear trs && wcr k trs =
      cpsCriterion tt emptyTRS k trs [[]]
  | otherwise = return MAYBE

-- CPS(R,C)
-- TODO: fix the worst case that C is not a proper subset of R
--       i.e. C is a renaming of R
--       possibly, canonical-TRS representation solves this
cps :: TRS -> TRS -> TRS
cps r c = nub
  [rule | (a, b) <- pair r,
           notElem (a, b) (pair c),
           (t, _, s, u) <- TRS.cpeak2 [a] [b],
           rule <- [(s,t),(s,u)]]
  where
    pair xs = [(a,b) | a <- xs, b <- xs]

joinable :: Int -> TRS -> (Term,Term) -> Bool
joinable k trs (s,t) =
  intersect
    (at_most_k_steps k trs s)
    (at_most_k_steps k trs t) /= []

wcr :: Int -> TRS -> Bool
wcr k trs = all (joinable k trs) (cp trs)
