module Closedness (
  -- Criteria
  strongClosednessCriterion,
  parallelClosednessCriterion,
  almostParallelClosednessCriterion,
  developmentClosednessCriterion,
  almostDevelopmentClosednessCriterion,
  toyama81,
  -- Predicates
  stronglyClosed,
  parallelClosed,
  almostParallelClosed,
  developmentClosed,
  almostDevelopmentClosed,
) where

import Data.List
import Term
import Rule
import TRS
import Rewriting
import PCP (pcp)
import Result
import qualified Proof


-- Strong closedness (Huet 1980)

stronglyJoinable :: Int -> TRS -> Term -> Term -> Bool
stronglyJoinable k trs s t =
  intersect
    (at_most_k_steps k trs s)
    (at_most_k_steps 1 trs t) /= []

stronglyClosed :: Int -> TRS -> Bool
stronglyClosed k trs = all p (cp trs)
  where
    p (s, t) =
      stronglyJoinable k trs s t &&
      stronglyJoinable k trs t s

strongClosednessCriterion :: Int -> Criterion
strongClosednessCriterion k trs
  | TRS.linear trs && stronglyClosed k trs =
    return (YES proof)
  | otherwise = return MAYBE
  where
    proof =
      Proof.proof
        "strong closedness (Huet 1980)"
        trs
        "The TRS R is linear and every critical pair is strongly closed."

-- Parallel closedness (Huet 1980)

parallelClosed :: TRS -> Bool
parallelClosed trs = all p (cp trs)
  where p (s, t) = elem t (parallelStep trs s)

parallelClosednessCriterion :: Criterion
parallelClosednessCriterion trs
  | TRS.leftLinear trs && parallelClosed trs =
    return (YES proof)
  | otherwise = return MAYBE
  where
    proof =
      Proof.proof
        "parallel closedness (Huet 1980)."
        trs
        "The TRS R is let-linear and every critical pair is parallel closed."

-- Almost parallel closedness (Toyama 1988)

cpWithPositions :: TRS -> [(Term, Position, Term)]
cpWithPositions trs =
  [ (replace (Term.substitute l2 sigma) (Term.substitute r1 sigma) p, p,
     Term.substitute r2 sigma)
  | rule2 <- trs,
    let (l2, r2) = Rule.rename "y" 1 rule2,
    rule1 <- trs,
    let (l1, r1) = Rule.rename "x" 1 rule1,
    p <- functionPositions l2,
    let l2p = subtermAt l2 p,
    Just sigma <- [mgu l2p l1],
    p /= [] || not (Rule.variant rule1 rule2) ]

almostParallelClosed1 :: Int -> TRS -> (Term, Position, Term) -> Bool
almostParallelClosed1 k trs (s, [], t) =
  intersect (at_most_k_steps k trs s) (parallelStep trs t) /= []
almostParallelClosed1 _ trs (s, _, t) =
  elem t (parallelStep trs s)

almostParallelClosed :: Int -> TRS -> Bool
almostParallelClosed k trs =
  all (almostParallelClosed1 k trs) (cpWithPositions trs)

almostParallelClosednessCriterion :: Int -> Criterion
almostParallelClosednessCriterion k trs
  | TRS.leftLinear trs && almostParallelClosed k trs =
    return (YES proof)
  | otherwise = return MAYBE
  where
    proof =
      Proof.proof
        "almost parallel closedness (Toyama 1988)"
        trs
        "The TRS R is left-linear and every critical pair is almost parallel closed."

-- Development closedness (van Oostrom 1997)

developmentClosed :: TRS -> Bool
developmentClosed trs = all p (cp trs)
  where p (s, t) = elem t (multistep trs s)

developmentClosednessCriterion :: Criterion
developmentClosednessCriterion trs
  | TRS.leftLinear trs && developmentClosed trs =
    return (YES proof)
  | otherwise = return MAYBE
  where
    proof =
      Proof.proof
        "development closedness (van Oostrom 1997)"
        trs
        "The TRS R is left-linear and every critical pair is development closed."

-- Almost development closedness (van Oostrom 1997)

almostDevelopmentClosed1 :: Int -> TRS -> (Term, Position, Term) -> Bool
almostDevelopmentClosed1 k trs (s, [], t) =
  intersect (at_most_k_steps k trs s) (multistep trs t) /= []
almostDevelopmentClosed1 _ trs (s, _, t) =
  elem t (multistep trs s)

almostDevelopmentClosed :: Int -> TRS -> Bool
almostDevelopmentClosed k trs =
  all (almostDevelopmentClosed1 k trs) (cpWithPositions trs)

almostDevelopmentClosednessCriterion :: Int -> Criterion
almostDevelopmentClosednessCriterion k trs
  | TRS.leftLinear trs && almostDevelopmentClosed k trs =
    return (YES proof)
  | otherwise = return MAYBE
  where
    proof =
      Proof.proof
        "almost development closedness (van Oostrom 1997)"
        trs
        "The TRS R is left-linear and every critical pair is almost development closed."


-- Toyama's criterion for parallel critical pairs (1981)
common :: [Term] -> [Term] -> Maybe Term
common ts us =
  case intersect ts us of
    [] -> Nothing
    v : _ -> Just v

closed1 :: Int -> TRS -> (Term, [String], Term) -> Maybe Term
closed1 k trs (t, xs, u) =
  common (at_most_k_steps k trs t)
         [ v | (_, v) <- restrictedParallelStep trs xs u ]

allClosed1 :: Int -> TRS -> [(Term, (TRS, Rule), [String], Term)] -> Maybe [(Term, Term, Term)]
allClosed1 _ _ [] = Just []
allClosed1 k trs ((t, _, xs, u) : peaks)
  | Just v <- closed1 k trs (t, xs, u),
    Just valleys <- allClosed1 k trs peaks =
    Just ((t, v, u) : valleys)
  | otherwise = Nothing

closed2 :: Int -> TRS -> (Term, Term) -> Maybe Term
closed2 k trs (t, u) =
  common (parallelStep trs t) (at_most_k_steps k trs u)

allClosed2 :: Int -> TRS -> [(Term, Term)] -> Maybe [(Term, Term, Term)]
allClosed2 _ _ [] = Just []
allClosed2 k trs ((t, u) : peaks)
  | Just v <- closed2 k trs (t, u),
    Just valleys <- allClosed2 k trs peaks =
    Just ((t, v, u) : valleys)
  | otherwise = Nothing

toyama81 :: Int -> Criterion
toyama81 k trs
  | TRS.leftLinear trs,
    Just valleys2 <- allClosed2 k trs (cp trs),
    Just valleys1 <- allClosed1 k trs (pcp trs) =
    return (YES (proof valleys1 valleys2))
  | otherwise = return MAYBE
  where
    proof v1 v2 =
      Proof.proof
        "parallel closedness based on parallel critical pair (Toyama 1981)"
        trs
        (unlines
          (["The TRS R is left linear and the parallel critical pairs <-||- -root-> are closed as follows:",
            ""] ++
           [ "  " ++ show t ++ " -||-> " ++ show v ++ " <-^* " ++ show u
           | (t, v, u) <- v1 ] ++
           ["",
            "The ordinary critical pairs <- -root-> are closed as follows:",
            "" ] ++
           [ "  " ++ show t ++ " ->^* " ++ show v ++ " <-||- " ++ show u
           | (t, v, u) <- v2 ]))

