module Main where

import Data.List
import TRS
import Result
import CompositionalCriteria
import Parser
import qualified RL
import qualified PRL
import qualified PRLCert
import CPS
import qualified PCPS
import qualified Closedness as CL
import qualified SN
import qualified Proof
import qualified NonConfluence
import System.Environment
import qualified XML
-- import qualified CeTA

type PP = Criterion -> Criterion

-- Menu
criterion :: String -> SN.RTermination -> String -> Int -> Criterion
criterion thm sn smt k = case thm of
  -- "rl"         -> RL.ruleLabeling smt k
  "prl"        -> PRL.parallelRuleLabeling smt k
  -- "cps"        -> CPS.hm11 sn k
  "pcps"       -> PCPS.hm11 sn k
  "sc"         -> CL.strongClosednessCriterion k
  "orthogonal" -> orthogonal
  "empty"      -> emptyTRS
  _            -> error emsg
  where
    emsg = unlines
      [ "unkown confluence criterion:" ++ thm
      , "supported criteria:"
      ,  unwords supportedCriteria ]

supportedCriteria :: [String]
supportedCriteria =
  -- ["rl", "prl", "cps", "pcps", "sc", "orthogonal", "empty"]
  ["prl", "pcps", "sc", "orthogonal", "empty"]

orthogonal :: Criterion
orthogonal trs
  | TRS.leftLinear trs && TRS.cp trs == []
    = return (YES proof)
  | otherwise
    = return MAYBE
  where
    proof =
      Proof.proof "orthogonality" trs "The left-linear and non-overlapping TRS is confluent"

cpsWith :: PP -> Criterion -> SN.RTermination -> String -> String -> Int -> Criterion
cpsWith pp noncr sn smt arg k =
  CPS.sh22 sn k (
    pp (apply (criterion arg sn smt k)))
  where
    apply cr = oneOf [noncr, cr]

pcpsWith :: PP -> Criterion -> SN.RTermination -> String -> String -> Int -> Criterion
pcpsWith pp noncr sn smt arg k =
  PCPS.sh22 sn k
    (pp (apply (criterion arg sn smt k)))
  where
    apply cr = oneOf [noncr, cr]

pcpcsWith :: PP -> Criterion -> SN.RTermination -> String -> String -> Int -> Criterion
pcpcsWith pp noncr sn smt arg k =
  parallelCriticalPairClosing k
      (pp (apply (criterion arg sn smt k)))
  where
    apply cr = oneOf [noncr, cr]

prlWith :: PP -> Criterion -> SN.RTermination -> String -> String -> Int -> Criterion
prlWith pp noncr sn smt arg k =
  PRL.sh22 smt k (
    pp (apply (criterion arg sn smt k)))
  where
    apply cr = oneOf [noncr, cr]


-- main
help :: IO ()
help = do
  putStrLn "hakusan [option1]* [option2] <file.trs>"
  putStrLn "(version 0.8)"
  putStrLn "Options 1:"
  putStrLn "  -tt <s>        uses termination tool <s> with WST format"
  putStrLn "  -ttx <s>       uses termination tool <s> with XML format"
  putStrLn "  -smt <s>       uses SMT solver <s>"
  putStrLn "  -ceta <s>      uses certifier CeTA <s>"
  putStrLn "  -noncr         enable non-confluence function"
  putStrLn "  -reduce <k>    enable reduction for each application of criterion (put after -smt)"
  putStrLn "Options 2:"
  putStrLn "  -orthogonal    orthogonality (Rosen 1973)"
  putStrLn "  -sc <k>        strong closedness (Huet 1980)"
  putStrLn "  -pc            parallel closedness (Huet 1980)"
  putStrLn "  -apc <k>       almost parallel closedness (Toyama 1988)"
  putStrLn "  -dc            development closedness (van Oostrom 1997)"
  putStrLn "  -adc <k>       almost development closedness (van Oostrom 1997)"
  putStrLn "  -toyama81 <k>  enhanced parallel closedness (Toyama 1981)"
  -- putStrLn "  -orl <k>       rule labeling with one labeling function (van Oostrom 2008)"
  putStrLn "  -rl <k>        rule labeling (van Oostrom 2008)"
  -- putStrLn "  -oprl <k>      parallel rule labeling with one labeling function (Zankl et al. 2015)"
  putStrLn "  -prl <k>       parallel rule labeling (Zankl et al. 2015)"
  putStrLn "  -prlcert <k>   parallel rule labeling (Zankl et al. 2015); outputs a certificate"
  putStrLn "  -cps <k>       critical pair system (Hirokawa and Middeldorp 2011)"
  putStrLn "  -pcpcs <k>     parallel-critical-pair-closing system"
  putStrLn "  -prl-<X> <k>   compositional parallel rule labeling with criterion <X>"
  -- putStrLn "  -cps-<X> <k>   compositional critical pair system with criterion <X>"
  putStrLn "  -pcps-<X> <k>  compositional parallel critical pair system with criterion <X>"
  putStrLn "  -pcpcs-<X> <k> compositional parallel-critical-pair-closing system with criterion <X>"
  putStrLn ("where <X> ::= " ++ intercalate " | " supportedCriteria)
  putStrLn ""
  putStrLn "When no option is specified, the next options are used:"
  putStrLn "'-noncr', '-reduce 5', '-prl-pcps 5', and '-pcps-prl 5'."

data Config = Config {
  _criterion :: Maybe Criterion,
  _smt :: String,
  _termination :: SN.Tool,
  _ceta :: String,
  _reduce :: Criterion -> Criterion,
  _nonCR :: Criterion
}

-- default strategy is equivalent to options "hakusan -reduce k -noncr -auto k"
defaultStrategy :: Int -> String -> SN.RTermination -> Criterion
defaultStrategy k smt tt trs =
  oneOf [noncr, red (auto k red smt tt)] trs
  where
    red = reduction smt k
    noncr = NonConfluence.nonWCR

auto :: Int -> (Criterion -> Criterion) -> String -> SN.RTermination -> Criterion
auto k pp smt tt =
  oneOf (autoCriteria k pp smt tt)

autoCriteria :: Int -> (Criterion -> Criterion) -> String -> SN.RTermination -> [Criterion]
autoCriteria k pp smt tt =
  [ PRL.sh22 smt k (pp (PCPS.hm11 tt k))
  , PCPS.sh22 tt k (pp (PRL.parallelRuleLabeling smt k))
  ]

parseArgs :: Config -> [String] -> IO ()
parseArgs c [file] = do
  spec <- readSpecFile file
  case spec of
    Left e    -> print e
    Right trs -> do
      result <- oneOf [noncr, thm] trs
      print result
  where
    thm   = case _criterion c of
      Just a  -> pp a
      Nothing -> defaultStrategy k smt tt
    k     = 5
    smt   = _smt c
    tt    = SN.sn (_termination c)
    pp    = _reduce c
    noncr = _nonCR c
parseArgs c ("-smt"   : s : args) =
  parseArgs (c { _smt = s }) args
parseArgs c ("-tt"    : s : args) =
  parseArgs (c { _termination = (SN.WST, s) }) args
parseArgs c ("-ttx"   : s : args) =
  parseArgs (c { _termination = (SN.XML, s) }) args
parseArgs c ("-ceta"  : s : args) =
  parseArgs (c { _ceta = s }) args
parseArgs c ("-reduce": s : args) =
  -- if -reduce option is added, do reduction step even if it is not confluence
  -- criterion (see the first case of parseArgs)
  parseArgs (c { _reduce = reduction (_smt c) k }) args
  where k = read s :: Int
parseArgs c ("-noncr" : args) =
  parseArgs (c { _nonCR = NonConfluence.nonWCR }) args
parseArgs c ("-orthogonal": args) =
  parseArgs (c { _criterion = Just orthogonal}) args
parseArgs c ("-sc"       : s : args) =
  parseArgs (c { _criterion = Just (CL.strongClosednessCriterion k)}) args
  where k = read s :: Int
parseArgs c ("-pc"           : args) =
  parseArgs (c { _criterion = Just CL.parallelClosednessCriterion}) args
parseArgs c ("-apc"      : s : args) =
  parseArgs (c { _criterion = Just (CL.almostParallelClosednessCriterion k)}) args
  where k = read s :: Int
parseArgs c ("-dc"           : args) =
  parseArgs (c { _criterion = Just CL.developmentClosednessCriterion }) args
parseArgs c ("-adc"      : s : args) =
  parseArgs (c { _criterion = Just (CL.almostDevelopmentClosednessCriterion k)}) args
  where k = read s :: Int
parseArgs c ("-toyama81" : s : args) =
  parseArgs (c { _criterion = Just (CL.toyama81 k)}) args
  where k = read s :: Int
parseArgs c ("-orl"      : s : args) =
  parseArgs (c { _criterion = Just (RL.ordinaryRuleLabeling (_smt c) k)}) args
  where k = read s :: Int
parseArgs c ("-rl"       : s : args) =
  parseArgs (c { _criterion = Just (RL.ruleLabeling (_smt c) k) }) args
  where k = read s :: Int
parseArgs c ("-prl"      : s : args) =
  parseArgs (c { _criterion = Just (PRL.parallelRuleLabeling (_smt c) k) }) args
  where k = read s :: Int
parseArgs c ("-oprl"     : s : args) =
  parseArgs (c { _criterion = Just (PRL.ordinaryParallelRuleLabeling (_smt c) k) }) args
  where k = read s :: Int
parseArgs c ("-cps"      : s : args) =
  parseArgs (c { _criterion = Just (CPS.hm11 (SN.sn (_termination c)) k) }) args
  where
    k = read s :: Int
parseArgs c ("-pcps"     : s : args) =
  parseArgs (c { _criterion = Just (PCPS.hm11 (SN.sn (_termination c)) k) }) args
  where
    k = read s :: Int
parseArgs c ("-pcpcs"    : s : args) =
  parseArgs (c { _criterion = Just (rSelfParallelCriticalPairClosing k pp noncr) }) args
  where
    k = read s :: Int
    noncr = _nonCR c
    pp = _reduce c
parseArgs _ ["-xml", file] = do
  spec <- readSpecFile file
  case spec of
    Left e    -> print e
    Right trs -> do
      putStr (XML.showProblem (trs, []))
parseArgs c ["-prlcert", sk, file] = do
  spec <- readSpecFile file
  case spec of
    Left e    -> print e
    Right trs -> do
      m <- PRLCert.parallelRuleLabeling (_smt c) ks trs
      case m of
        Nothing -> putStrLn "MAYBE"
        Just certificate -> do
          putStrLn "YES"
          putStr (XML.showCertificate certificate)
          {-
          r <- CeTA.certify (_ceta c) certificate
          case r of
            CeTA.CERTIFIED -> do
              putStrLn "YES"
              putStr (XML.showCertificate certificate)
            CeTA.REJECTED (s, e) -> do
              putStrLn "ERROR"
              putStrLn "-- Certificate --"
              putStr (XML.showCertificate certificate)
              putStrLn "-- CeTA Output --"
              putStrLn s
              putStrLn "-- Parser Error --"
              putStrLn (show e)
          -}
  where 
    k = read sk ::Int
    ks = if k < 2 then [k] else [2,k]
-- compositional criteria options
parseArgs c (opt : s : args)
  | "-prl-"   `isPrefixOf` opt = continue prlWith "-prl-"
  | "-cps-"   `isPrefixOf` opt = continue cpsWith "-cps-"
  | "-pcps-"  `isPrefixOf` opt = continue pcpsWith "-pcps-"
  | "-pcpcs-" `isPrefixOf` opt = continue pcpcsWith "-pcpcs-"
  where
    continue thm lab =
      parseArgs
        (c { _criterion = Just (thm pp noncr tt smt (cutOpt lab) k)})
        args
    cutOpt h = deleteFirstsBy (==) opt h
    k        = read s :: Int
    tt       = SN.sn (_termination c)
    smt      = _smt c
    pp       = _reduce c
    noncr    = _nonCR c
parseArgs _ _ = help

defaultConfig :: Config
defaultConfig = Config {
  _criterion   = Nothing,
  _smt         = "z3",
  _termination = (SN.XML, "NaTT.exe"),
  _ceta        = "ceta",
  _reduce      = id,
  _nonCR       = failTheorem
}

main :: IO ()
main = do
  args <- getArgs
  parseArgs defaultConfig args
