module Proof where

import TRS (TRS)
import qualified TRS as TRS

-- An example of proofs:
-- ==========================
-- # Parallel-critical-pair-closing system (Shintani and Hirokawa 2022).
-- 
-- Consider the left-linear TRS R:
-- 
--   (empty)
-- 
-- Let C be the following subset of R:
-- 
--   (empty)
-- 
-- Then C is parallel-critical-pair-closing for R.
-- The confluence of R follows from that of C.
-- <sub proof>
-- ==========================
-- This is generated by
--   subproof
--     "Parallel-critical-pair-closing system (Shintani and Hirokawa 2022)."
--     []
--     []
--     "Then C is parallel-critical-pair-closing for R."
--     "<sub proof>"

data ProofBlock = PB
  { method    :: String
  , caption   :: String
  , system    :: TRS
  , proofText :: String
  } deriving Show

defaultPB :: ProofBlock
defaultPB = PB
  "<method>"
  "Consider the left-linear TRS R:"
  []
  "<proof>"

proof :: String -> TRS -> String -> String
proof m trs t = proofB $ defaultPB
  { method = m
  , system = trs
  , proofText = t }

subproof :: String -> TRS -> TRS -> String -> String -> String
subproof m trs subtrs t1 t2 = proofB $ defaultPB
  { method = m
  , system = trs
  , proofText = unlines
    [ "Let C be the following subset of R:"
    , ""
    , showTRS subtrs
    , t1 ++ "Therefore, the confluence of R follows from that of C."
    , ""] ++
    t2
  }

subeqproof :: String -> TRS -> TRS -> String -> String -> String
subeqproof m trs subtrs t1 t2 = proofB $ defaultPB
  { method = m
  , system = trs
  , proofText = unlines
    [ "Let C be the following subset of R:"
    , ""
    , showTRS subtrs
    , t1 ++ "Therefore, the confluence of R is equivalent to that of C."
    , ""] ++
    t2
  }

splitproof :: String -> TRS -> (TRS,TRS) -> String -> String -> String
splitproof m trs (a,b) t1 t2 = proofB $ defaultPB
  { method = m
  , system = trs
  , proofText = unlines
    [ "Let S1 and S2 be a decomposition of R:"
    , ""
    , showTRS a
    , "and"
    , ""
    , showTRS b
    , t1 ++ "Therefore, the confluence of R follows from that of S1 and S2."
    , ""] ++
    t2
  }


proofB :: ProofBlock -> String
proofB (PB m c trs p) = 
  unlines [ "# " ++ m
          , ""
          , c
          , ""
          , showTRS trs ] ++
  p

showTRS :: TRS -> String
showTRS []  = "  (empty)\n"
showTRS trs = TRS.showTRS trs
