module Result where

import Data.List (find)
import TRS

data Result =
    YES String
  | NO  String
  | MAYBE

instance Show Result where
  show (YES s) = "YES\n\n" ++ s
  show (NO s)  = "NO\n\n" ++ s
  show MAYBE   = "MAYBE\n"

type Criterion = TRS -> IO Result

succedTheoremYES :: Criterion
succedTheoremYES trs = return (YES proof)
  where
    proof = "always YES:\n" ++ showTRS trs

succedTheoremNO :: Criterion
succedTheoremNO trs = return (NO proof)
  where
    proof = "always NO:\n" ++ showTRS trs

failTheorem :: Criterion
failTheorem _ = return MAYBE

mapResult :: (String -> String) -> Result -> Result
mapResult f (YES s) = YES (f s)
mapResult f (NO s)  = NO (f s)
mapResult _ MAYBE   = MAYBE

isYES, isNO :: Result -> Bool
isYES (YES _) = True
isYES _       = False
isNO (NO _)   = True
isNO _        = False

yesWhen, noWhen :: Bool -> Result
yesWhen True  = YES "yes"
yesWhen False = MAYBE
noWhen True   = NO "no"
noWhen False  = MAYBE

lookupYES :: [Result] -> Maybe Result
lookupYES rs = find isYES rs

lookupNO :: [Result] -> Maybe Result
lookupNO rs = find isNO rs

lookupResult :: [Result] -> Maybe Result
lookupResult rs
  | Just x <- lookupYES rs = Just x
  | Just x <- lookupNO rs  = Just x
  | otherwise              = Nothing

anyYES, anyNO :: (a -> Result) -> [a] -> Bool
anyYES f xs =
  case lookupYES (map f xs) of
    Just _ -> True
    Nothing -> False

anyNO f xs =
  case lookupNO (map f xs) of
    Just _ -> True
    Nothing -> False

-- find first non-MAYBE result from criteria
oneOf :: [Criterion] -> Criterion
oneOf [] _ = return MAYBE
oneOf (thm:criteria) trs = do
  v <- thm trs
  case v of
    MAYBE -> oneOf criteria trs
    _ -> return v

