module XML where

import Data.List
import Term
import Rule
import TRS

type Attribute = (String, String)

data XML =
    Tag String [Attribute] [XML]
  | Text String
  | Int Int

type Labeling = [(Rule, Int)]

data Proof = 
    EmptyTRS
  | Composition Labeling Labeling Int TRS Proof 

type Certificate = (TRS, Proof)

escape :: String -> String
escape ""         = ""
escape ('&'  : cs) = "&amp;"  ++ escape cs
escape ('<'  : cs) = "&lt;"   ++ escape cs
escape ('>'  : cs) = "&gt;"   ++ escape cs
escape ('\'' : cs) = "&apos;" ++ escape cs
escape ('"'  : cs) = "&quot;" ++ escape cs
escape (c    : cs) = c : escape cs

-- name="value"
showAttribute :: Attribute -> String
showAttribute (name, value) = name ++ "=" ++ show value

openTag :: String -> [Attribute] -> String
openTag name as = 
  "<" ++ intercalate " " (name : [ showAttribute a | a <- as ]) ++ ">"

closeTag :: String -> String
closeTag name = "</" ++ name ++ ">"

instance Show XML where
  show (Int n)            = show n
  show (Text s)           = escape s
  show (Tag name as xmls) =
    openTag name as ++ "\n" ++
    concat [ show xml | xml <- xmls ]  ++ 
    closeTag name ++ "\n"

tag :: String -> [XML] -> XML
tag name xmls = Tag name [] xmls

tag1 :: String -> XML -> XML
tag1 name xml = tag name [xml]

tagFuncSym :: (String, Int) -> XML
tagFuncSym (f, n) = 
  tag "funcsym" [tag1 "name" (Text f), tag1 "arity" (Int n)]

tagSignature :: Signature -> XML
tagSignature sig =
  tag "signature" [ tagFuncSym fn | fn <- sig ]

tagTerm :: Term -> XML
tagTerm (V x)    = tag1 "var" (Text x)
tagTerm (F f ts) = 
  tag "funapp" 
    (tag1 "name" (Text f) :
     [ tag "arg" [tagTerm t] | t <- ts ])

tagRule :: Rule -> XML
tagRule (l, r) = 
  tag "rule" [tag "lhs" [tagTerm l], 
              tag "rhs" [tagTerm r]]

tagRelrules :: TRS -> XML
tagRelrules trs = 
  tag "relrules" [ tagRule lr | lr <- trs ]

tagRules :: (TRS, TRS) -> XML
tagRules (rs, []) = 
  tag "rules" [ tagRule lr | lr <- rs ]
tagRules (rs, ss) = 
  tag "rules" ([ tagRule lr | lr <- rs ] ++ [tagRelrules ss])

tagRelativeTRS :: (TRS, TRS) -> XML
tagRelativeTRS (rs, ss) = 
  tag "trs" [tagRules (rs, ss), 
             tagSignature (TRS.signatureOf (rs ++ ss))]

tagProblem :: (TRS, TRS) -> XML
tagProblem (rs, ss) = 
  Tag "problem" 
      [("xmlns:xsi","http://www.w3.org/2001/XMLSchema-instance"),
       ("type","termination"),
       ("xsi:noNamespaceSchemaLocation","/xtc.xsd")]
      [tagRelativeTRS (rs, ss),
       tag1 "strategy" (Text "FULL")]

tagCpfVersion :: String -> XML
tagCpfVersion s = tag1 "cpfVersion" (Text s)

tagRuleLabelingFunctionEntry :: (Rule, Int) -> XML
tagRuleLabelingFunctionEntry (rule, n) =
  tag "ruleLabelingFunctionEntry"
      [tagRule rule, tag1 "label" (Int n)]

tagRuleLabelingFunction :: Labeling -> XML
tagRuleLabelingFunction phi =
  tag "ruleLabelingFunction"
      [ tagRuleLabelingFunctionEntry (rule, n) | (rule, n) <- phi ]

tagJoinsRS :: Int -> XML
tagJoinsRS k = tag1 "joinsRS" (tag1 "auto" (Int k))

tagCrProof' :: Proof -> XML
tagCrProof' EmptyTRS = 
  tag "orthogonal" []
tagCrProof' (Composition phi psi k trs proof) =
  tag "compositionalPcpRuleLabeling"
    [tagRuleLabelingFunction phi,
     tagRuleLabelingFunction psi,
     tagJoinsRS k,
     tagTRS trs,
     tagCrProof proof]

tagCrProof :: Proof -> XML
tagCrProof proof = tag1 "crProof" (tagCrProof' proof)

showProblem :: (TRS, TRS) -> String
showProblem (rs, ss) = 
  "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n" ++
  show (tagProblem (rs, ss))

tagTRS :: TRS -> XML
tagTRS trs = tag1 "trs" (tagRules (trs, []))

tagInput :: TRS -> XML
tagInput trs =
  tag1 "input" (tag1 "trsInput" (tagTRS trs))

tagCertificationProblem :: Certificate -> XML
tagCertificationProblem (trs, proof) = 
  Tag "certificationProblem"
      [("xmlns:xsi","http://www.w3.org/2001/XMLSchema-instance"),
       ("xsi:noNamespaceSchemaLocation", "cpf.xsd")]
      [tagInput trs,
       tag1 "cpfVersion" (Text "2.1"),
       tag1 "proof" (tagCrProof proof)]

showCertificate :: Certificate -> String
showCertificate certificate =
  "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n" ++
  "<?xml-stylesheet type=\"text/xsl\" href=\"cpfHTML.xsl\"?>\n" ++
  show (tagCertificationProblem certificate)
