(* Rule labeling based on conversions *)
open Minismt
open Term
open Substitution
open Overlap
open Wcr
open Format

let minimal ~geq xs = 
  Listx.unique [ x | x <- xs; List.for_all (fun y -> not (geq x y) || x = y) xs ]

(*
let minimal ~geq xs = xs
*)

let rec emb xs ys =
  match xs, ys with
  | _, [] -> true
  | [], _ :: _ -> false
  | x :: xs, y :: ys -> (x = y && emb xs ys) || emb xs (y :: ys)

let pairwise_emb (xs, ys) (xs', ys') = emb xs xs' && emb ys ys'

let rec lookup rule = function
  | [] -> failwith "Rlc.lookup"
  | (rule', x) :: _ when Rule.variant rule rule' -> x
  | _ :: a -> lookup rule a

let rec rewrite_root ix s = 
  [ x, substitute (pattern_match l s) r 
  | (l, r), x <- ix; is_instance_of s l ]

let rewrite ix s =
  Listx.unique
    [ x, replace s t p 
    | p <- positions s; x, t <- rewrite_root ix (subterm_at p s) ]

(* X^(k) *)
let rec jj1 ix k ss =
  if k = 0 then ss else 
  minimal ~geq:(fun (xs1, s1) (xs2, s2) -> s1 = s2 && emb xs1 xs2) 
    (ss @ 
     jj1 ix (k - 1) 
       [ xs @ [x], t | xs, s <- ss; x, t <- rewrite ix s ])
	
let jj ix k (s, t) = 
  let l1 = jj1 ix k [[], s] 
  and l2 = jj1 ix k [[], t] in
  minimal ~geq:pairwise_emb [ xs, ys | xs, s' <- l1; ys, t' <- l2; s' = t' ]

let psi ix rules a b = function
  | [] -> C []
  | c :: ds ->
      let ord =
        if List.mem c [ c' |  rule, c' <- ix; List.mem rule rules ] then
          (fun x y -> Eq (x, y)) 
        else  
          (fun x y -> Gt (x, y))
      in
      C (ord b c :: [ D [Gt (a, d); Gt (b, d)] | d <- ds])

let phi_ab ix rules a b cs =
  D [ C (psi ix rules a b cs2 :: [ Gt (a, c) | c <- cs1 ]) 
    | cs1, cs2 <- List.combine (Listx.prefix cs) (Listx.suffix cs) ]

let phi ix rules k ((rule1, p, rule2, mu) as o) =
  D [ C [phi_ab ix rules (lookup rule1 ix) (lookup rule2 ix) cs;
         phi_ab ix rules (lookup rule2 ix) (lookup rule1 ix) ds]
    | cs, ds <- jj ix k (cp_of_overlap o) ]

let rl ix rules k = 
  C [ phi ix rules k o | o <- overlap rules ]

(*
let inverse rules = 
  [ r, l | l, r <- rules; Listset.subset (variables l) (variables r) ]
*)
let inverse rules = 
  [ r, l | l, r <- rules; 
           not (Term.is_variable r) &&
           Listset.subset (variables l) (variables r) ]

let rec unique ~eq = function
  | [] -> []
  | x :: ys -> x :: unique ~eq [ y | y <- ys; not (eq x y) ]

let solve ~tool k rules =
  if Rules.linear rules then
    let inv = inverse rules in
    let union = unique ~eq:Rule.variant (Listset.union rules inv) in
    let ix = [ rule, sprintf "x%d" n | n, rule <- Listx.ix ~i:1 union ] in
    match 
      Minismt.sat ~tool [ x | _, x <- ix] 
        (C (rl ix rules k :: [ Eq (lookup (l, r) ix, lookup (r, l) ix) | l, r <- inv ]))
    with
    | Some b -> Some [ rule, List.assoc x b | rule, x <- ix; List.mem rule rules ]
    | None -> None
  else 
    None
