open Format

type t = Term.t * Term.t

let print ppf (l, r) =
  fprintf ppf "@[<2>%a ->@ %a@]" Term.print l Term.print r

let print_with sep ppf (l, r) =
  fprintf ppf "@[<2>%a %s@ %a@]" Term.print l sep Term.print r

let equal = (=)

let hash = Hashtbl.hash 

let compare = Pervasives.compare 

let variables (l, r) =
  Listx.unique (Term.variables l @ Term.variables r)

let functions (l, r) =
  Listx.unique (Term.functions l @ Term.functions r)

let signature (l, r) =
  Listx.unique (Term.signature l @ Term.signature r)


let is_non_erasing (l, r) =
  Listset.subset (Term.variables l) (Term.variables r)

let is_non_duplicating (l, r) =
  let p x = Term.count_variable x l >= Term.count_variable x r in
  List.for_all p (variables (l, r))

let variable_condition (l, r) =
  Listset.subset (Term.variables r) (Term.variables l)

let wf (l, r) =
  not (Term.is_variable l) && variable_condition (l, r)

let rename (l, r) =
  let s = [ x, Term.V (Term.fresh ()) | x <- Term.variables l ] in
  (Term.substitute s l, Term.substitute s r)

let left_linear (l, r) = Term.linear l 

let right_linear (l, r) = Term.linear r

let linear (l, r) = Term.linear l && Term.linear r

let variant (l1, r1) (l2, r2) =
  let lr1 = Term.F ("->", [l1; r1]) 
  and lr2 = Term.F ("->", [l2; r2]) in
  Substitution.is_instance_of lr1 lr2 &&
  Substitution.is_instance_of lr2 lr1

let rec remove rules rl =
 match rules with
  | [] -> []
  | st :: stt -> 
           if variant rl st 
           then remove stt rl
           else st :: (remove stt rl)
