package evotech;

import uchicago.src.sim.engine.*;
import uchicago.src.sim.gui.*;
import uchicago.src.sim.network.*;
import uchicago.src.sim.space.*;
import uchicago.src.sim.analysis.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;
import javax.swing.*;

import evotech.tools.*;

/**
 * <p>Title: EvoTech</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: </p>
 *
 * The graphical extension of the model with the 'observational' tools.
 *
 * @author Laszlo Gulyas & George Kampis
 * @version 1.0
 */
public class ModelGUI extends Model {

// Graphical gadgets
DisplaySurface dsurf;

OpenSequenceGraph numGraph;
OpenSequenceGraph slotGraph;

boolean isClustering;
OpenSequenceGraph speciesGraph;

boolean isSpeciesNum;
Plot  speciesPlot;

boolean isSpeciesDump;
PrintWriter speciesDump;

boolean isNetwork;
DisplaySurface netSurf;
ArrayList netList;
GraphLayout netLayout;
HashMap nodeMap;

boolean isEnergyGraph;
OpenSequenceGraph energyGraph;

boolean isAgeHistogram;
Histogram ageHistogram;

boolean isAvgDistanceGraph;
OpenSequenceGraph avgDistanceGraph;

// Variables
boolean isRunning;

// Statistics for average distance
int numDistances;
double cumulDistance;

// Property statistics (histogram)
boolean isPropertyHistogram;
OpenHistogram2 propertyHistogram;
//Histogram propertyHistogram;

// Property statistics (histogram)
boolean isPropertyHistogram2;
OpenHistogram2 propertyHistogram2;

////////////////////////////////////////////////////////////////////////////////

    protected void generateAgentsInABall(int num, double maxDist, int[] props) {

//      System.out.println("Entering...");

      Agent center = new Agent(idCount, this);
      idCount--;
//      System.out.println("  Center created.");

      center.props = new ArrayList();
      for (int i=0; i<numProps; i++) {
        center.props.add(new Integer(props[i]));
      }
//      System.out.println("  Center positioned.");

//      System.out.println("  Starting loop...");
      for (int i=0; i<num; i++) {
//        System.out.println("    #"+i);
        Agent anAgent;
        double d;
        do {
          anAgent = new Agent(idCount, this);
          d = center.compareProps(anAgent);
          if (d > maxDist) {
            idCount--;
          }
//          System.out.println("      Created. Dist="+d+" (maxDist="+maxDist+").");
        } while (d > maxDist);

        addAgent(anAgent);
//        System.out.println("      Added.");
      }
//      System.out.println("  Loop finished.");

//      System.out.println("Returning...");

//      updateGraphs();
    }

    protected void deleteAllAgents() {
      Iterator i = agentList.iterator();
      while (i.hasNext()) {
        Agent a = (Agent) i.next();
        if (isNetwork) {
          removeNode(a);
        }
        removeAgent(a);
      }

//      updateNetwork();
//      updateGraphs();
    }

  public void setup() {
    super.setup();

    isRunning = false;

    isClustering  = false;
    isSpeciesNum  = false;
    isSpeciesDump = false;
    isNetwork = false;
    isEnergyGraph = false;
    isAgeHistogram = false;
    isAvgDistanceGraph = false;
    isPropertyHistogram = true;
    isPropertyHistogram2 = true;

    name = "EvoTech Model II. v0.11";

    // Making sure the parameters show up in the order we put them in
    // and not in an alphabetical order.
    Controller.ALPHA_ORDER = false;
    params = new String[] {
               "numAgents",
               "numProps",
               "minProp", "maxProp",
               "pEncounter", "pCrossOver", "pMutation", "pNewSlot",
               "mappingLimit", "mappingMax",
               "isDeath",
               "eConsumption", "eInput", "eDisc",
               "initEnergy", "incrEnergy",
               "isClustering",
               "isSpeciesNum",
               "isSpeciesDump",
               "isNetwork",
               "isEnergyGraph",
               "isAgeHistogram",
               "isAvgDistanceGraph",
               "isPropertyHistogram",
               "isPropertyHistogram2"
             };

    if (dsurf != null) {  // remove the old display, if new run...
      dsurf.dispose();
      dsurf = null;
    }

    if (numGraph != null) { // remove the old graph, if new run...
      numGraph.dispose();
      numGraph = null;
    }

    if (slotGraph != null)  { // remove the old graph, if new run...
      slotGraph.dispose();
      slotGraph = null;
    }

    if (speciesGraph != null) { // remove the old histogram, if new run
      speciesGraph.dispose();
      speciesGraph = null;
    }

    if (speciesPlot != null) {
      speciesPlot.dispose();
      speciesPlot = null;
    }

    if (netSurf != null) {
      netSurf.dispose();
      netSurf = null;
    }

    if (energyGraph != null) {
      energyGraph.dispose();
      energyGraph = null;
    }

    if (ageHistogram != null) {
      ageHistogram.dispose();
      ageHistogram = null;
    }

    if (avgDistanceGraph != null) {
      avgDistanceGraph.dispose();
      avgDistanceGraph = null;
    }

    if (propertyHistogram != null) {
      propertyHistogram.dispose();
      propertyHistogram = null;
    }

    if (propertyHistogram2 != null) {
      propertyHistogram2.dispose();
      propertyHistogram2 = null;
    }

    // System cleaning up
    System.gc();

    Controller cont = (Controller) getController();

    JButton jb = new JButton("Delete");
    jb.setToolTipText("Deletes the population.");
    jb.addActionListener( new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        deleteAllAgents();
      }
    });
    cont.addButton(jb);

    final ModelGUI m = this;
    jb = new JButton("Generate");
    jb.setToolTipText("Generates Agents 'in a Ball'");
    jb.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        BallGeneratorFrame f = new BallGeneratorFrame(m);
        f.show();
      }
    });
    cont.addButton(jb);
  }


    protected void createNetDisplay() {
      netSurf = new DisplaySurface(this, "Network");
      netLayout = new FruchGraphLayout(netList, 600, 600, netSurf, 0);
//      netLayout = new CircularGraphLayout(netList, 400, 400);
      Network2DDisplay netDisplay = new Network2DDisplay(netLayout);
      netSurf.addDisplayableProbeable(netDisplay, "Network Representation");
      addSimEventListener(netSurf);
      netSurf.display();
    }

    protected void createAgeHistogram() {
      ageHistogram = new Histogram("Age Distribution", 31, 0, 30, this);
      ageHistogram.createHistogramItem("Age", agentList, new BinDataSource() {
                                         public double getBinValue(Object o) {
                                           if (o instanceof Agent) {
                                             Agent a = (Agent) o;
                                             return (double) a.age;
                                           } else {
                                             return 0;
                                           }
                                         }
                                       }
      );
      this.registerMediaProducer("AgeHistogram", ageHistogram);
      ageHistogram.display();
    }

    protected void createAvgDistanceGraph() {
      avgDistanceGraph = new OpenSequenceGraph("Average Agent-Agent Distance", this);
      avgDistanceGraph.addSequence( "Avg Sequence", new Sequence() {
                                      public double getSValue() {
                                        return cumulDistance / (double) numDistances;
                                      }
                                    }
      );
      avgDistanceGraph.setYRange(0, 1);
      this.registerMediaProducer("AvgDistanceGraph", avgDistanceGraph);
      avgDistanceGraph.display();
    }

      protected void createPropertyHistogramItem(int propertyNum) {
        final int currentProp = propertyNum;
        propertyHistogram.createHistogramItem( "Prop #"+currentProp,
                                               agentList,
                                               new BinDataSource() {
            public double getBinValue(Object o) {
              Agent a = (Agent) o;
              if (a.props.size() <= currentProp) {
                return minProp - 1;
              } else {
                return ((Integer)a.props.get(currentProp)).doubleValue();
              }
            }
          }
        );
      }

    protected void createPropertyHistogram() {
/*
      propertyHistogram = new Histogram("Distribution of Properties",
                                             (maxProp-minProp+1),
                                             minProp, maxProp+0.5,
                                             this );
*/
      propertyHistogram = new OpenHistogram2( "Distribution of Properties",
                                             (maxProp-minProp+1),
                                             minProp, maxProp+1,
                                             this );

      propertyHistogram.setAxisTitles("Value", "# of Agents");

      for (int i=0; i<numProps; i++) {
        createPropertyHistogramItem(i);
      }
      this.registerMediaProducer("PropertyHistogram", propertyHistogram);
      propertyHistogram.display();
    }

      protected void createPropertyHistogram2Item(int propertyNum) {
        final int currentProp = propertyNum;
        propertyHistogram2.createHistogramItem( "Value"+currentProp,
                                               valueList[currentProp],
                                               new BinDataSource() {
            public double getBinValue(Object o) {
              return ((Integer) o).doubleValue();
            }
          }
        );
      }

    ArrayList[] valueList;

    protected void createPropertyHistogram2() {
      propertyHistogram2 = new OpenHistogram2( "Distribution of Properties 2",
                                             numProps,
                                             0, numProps,
                                             this );

      propertyHistogram2.setAxisTitles("Property", "# of Agents");

      valueList = new ArrayList[maxProp-minProp+1];
      for (int i=minProp; i<=maxProp; i++) {
        valueList[i-minProp] = new ArrayList();
        createPropertyHistogram2Item(i);
      }
      this.registerMediaProducer("PropertyHistogram2", propertyHistogram2);
      propertyHistogram2.display();
    }

  public void buildModel() {

    isRunning = true;

    if (isNetwork) {
      netList = new ArrayList();
      nodeMap = new HashMap();
    }

    super.buildModel();

    // Creating the time series
    slotGraph = new OpenSequenceGraph("Number of slots", this);
    slotGraph.addSequence("num", new Sequence() {
                             public double getSValue() {
                               return (double) numProps;
                             }
                           }
                         );
    this.registerMediaProducer("NumberOfSlots", slotGraph);
    slotGraph.display();

    if (isClustering) {
      // Creating the histogram
      speciesGraph = new OpenSequenceGraph("Number of species", this);
      speciesGraph.addSequence("Num", new Sequence() {
                                     public double getSValue() {
                                       return (double) speciesList.size();
                                     }
                                   }                                  );
      calcClusters();
      this.registerMediaProducer("NumberOfSpecies", speciesGraph);
      speciesGraph.display();

      // Technically, we cannot do this without clustering, so it goes in here...
      if (isSpeciesNum) {
        speciesPlot = new Plot("The size of the species");
        speciesPlot.setAxisTitles("Species", "Population");
        speciesPlot.setConnected(false, 0);
        speciesPlot.setBars(1.0, 1.0);
        this.registerMediaProducer("SpeciesNum", speciesPlot);
        speciesPlot.display();
      }

      // Technically, we cannot do this without clustering, so it goes in here...
      if (isSpeciesDump) {
        try {
          speciesDump = new PrintWriter(new FileWriter("species.txt"));
        } catch (IOException ioe) {
          System.out.println("SYSTEM ERROR: Could not create file. "+ioe);
        }
      }
    }


    numGraph = new OpenSequenceGraph("Number of agents", this);
    numGraph.addSequence("num", new Sequence() {
                             public double getSValue() {
                               return (double) agentList.size();
                             }
                           }
                         );
    this.registerMediaProducer("NumberOfAgents", numGraph);
    numGraph.display();

    if (isNetwork) {
      createNetDisplay();

      if (isAvgDistanceGraph) {
        createAvgDistanceGraph();
      }
    }

    if (isEnergyGraph) {
      energyGraph = new OpenSequenceGraph("Available Energy in the World", this);
      energyGraph.addSequence("energy", new Sequence() {
                               public double getSValue() {
                                 return energy;
                               }
                             }
                           );
      this.registerMediaProducer("AvailableEnergy", energyGraph);
      energyGraph.display();
    }

    if (isAgeHistogram) {
      createAgeHistogram();
    }

    if (isPropertyHistogram) {
      createPropertyHistogram();
    }

    if (isPropertyHistogram2) {
      createPropertyHistogram2();
    }
  }

////////////////////////////////////////////////////////////////////////////////

  public void preStep() {
//    System.out.println("Tick #"+this.getTickCount());
  }

////////////////////////////////////////////////////////////////////////////////
// 'Magic' to calculate clusters of agents that can mate with each-other.
////////////////////////////////////////////////////////////////////////////////

    HashMap agentToSpecies;
    Species from, to;
    ArrayList speciesList;

      /**
       * Picks the minimum cluster-cluster distance and puts the
       * two species into 'from' and 'to' above, respectively.
       * @return The minimum species-to-species distance.
       */
      public double getMinDistance() {
        double min = Double.MAX_VALUE;
        Iterator i = speciesList.iterator();
        while (i.hasNext()) {
          Species s1 = (Species) i.next();
          Species s2 = s1.getClosest();
          if (s1.getDistanceTo(s2) < min) {
            from = s1;
            to = s2;
            min = s1.getDistanceTo(s2);
          }
        }

        return min;
      }

    /**
     * Calculates clusters of agents that can mate with each other.
     */
    public void calcClusters() {
      agentToSpecies = new HashMap();
      speciesList = new ArrayList();
      Iterator i = agentList.iterator();
      while (i.hasNext()) {
        Agent a = (Agent) i.next();
        speciesList.add(new Species(a));
      }

      i = speciesList.iterator();
      while (i.hasNext()) {
        Species s = (Species) i.next();
        s.calcInitDistances();
      }

      while (speciesList.size() > 1) {
        // If they cannot mate anymore, we quit
        if (getChildNum(getMinDistance(), 0) < 1)
//        if (getMinDistance() > 0.5)
          break;


        from.mergeWith(to);
        speciesList.remove(to);

        i = speciesList.iterator();
        while (i.hasNext()) {
          Species s = (Species) i.next();
          if (s != from)
            s.mergedAtoB(to, from);
        }
      }

    }

    public void updateSpeciesPlot() {
      speciesPlot.clear(0);
      for (int i=0; i<speciesList.size(); i++) {
        Species s = (Species) speciesList.get(i);
        speciesPlot.plotPoint(i, s.getSize(), 0);
      }

      speciesPlot.fillPlot();
    }


    public void dumpSpecies() {
      speciesDump.println("Tick #"+getTickCount());
      Iterator i = speciesList.iterator();
      while (i.hasNext()) {
        Species s = (Species) i.next();
        speciesDump.println(s);
      }
      speciesDump.println();
    }

////////////////////////////////////////////////////////////////////////////////

    protected void updateNetwork() {
      if (isNetwork) {
        Iterator i = removeList.iterator();
        while (i.hasNext()) {
          removeNode((Agent) i.next());
        }
      }
    }

    protected void updateGraphs() {
      numGraph.step();
      slotGraph.step();

      if (isClustering) {
        calcClusters();
        speciesGraph.step();
        if (isSpeciesNum) {
          updateSpeciesPlot();
        }
        if (isSpeciesDump) {
          dumpSpecies();
        }
      }

      if (isNetwork) {
        netLayout.setList(netList);
        if (netList.size() > 0) {
          netLayout.updateLayout();
        }
        netSurf.updateDisplay();

        if (isAvgDistanceGraph) {
          avgDistanceGraph.step();
        }
      }

      if (isEnergyGraph) {
        energyGraph.step();
      }

      if (isAgeHistogram) {
        ageHistogram.step();
      }

      if (isPropertyHistogram) {
        propertyHistogram.step();
      }

      if (isPropertyHistogram2) {
        for (int i=minProp; i<=maxProp; i++) {
          valueList[i-minProp].clear();
        }

        Iterator i = agentList.iterator();
        while (i.hasNext()) {
          Agent a = (Agent) i.next();
          Iterator j = a.props.iterator();
          int pos = 0;
          while (j.hasNext()) {
            Integer value = (Integer) j.next();
            int v = value.intValue();
            valueList[v].add(new Integer(pos));
            pos++;
          }
        }
        propertyHistogram2.step();
      }
    }

  public void postStep() {
    updateNetwork();
    super.postStep();
    updateGraphs();
  }

  public void atEnd() {
    if (isClustering) {
      if (isSpeciesDump) {
        speciesDump.close();
      }
    }
  }

////////////////////////////////////////////////////////////////////////////////
// Hacking for networks
////////////////////////////////////////////////////////////////////////////////

    protected void createNode(Agent a) {
      OvalNetworkItem item = new OvalNetworkItem(0, 0);
      DefaultDrawableNode aNode = new DefaultDrawableNode(item);
      aNode.setNodeLabel(a.ID+"");
      netList.add(aNode);
      nodeMap.put(a, aNode);

      for (int i=0; i<agentList.size(); i++) {
        Agent b = (Agent) agentList.get(i);
        if (b != a) {
          Node bNode = (Node) nodeMap.get(b);
          if (bNode != null) {  // If null, it's been removed in this round
            double dist = a.compareProps(b);

            // We calculate it, independent of isAvgDistanceGraph,
            // because it would be impossible to produce the correct
            // value after a period with no calculations when isNetwork=true.
            cumulDistance += dist;
            numDistances++;

            int num = getChildNum(dist, 0);

            if (num > 0) {
//              DefaultDrawableEdge edge = new DefaultDrawableEdge(aNode, bNode, "", num);
              if (dist == 0)
                dist = 0.00001;
              DefaultDrawableEdge edge = new DefaultDrawableEdge(aNode, bNode, "", (float) (1/dist));
              aNode.addOutEdge(edge);
              bNode.addInEdge(edge);
            }
          }
        }
      }
    }

    // For DEBUGGING!!!!
    public void markParents(Agent a, Agent p1, Agent p2) {
       Node aNode = (Node) nodeMap.get(a);
       Node p1Node = (Node) nodeMap.get(p1);
       Node p2Node = (Node) nodeMap.get(p2);
       DefaultDrawableEdge edge = new DefaultDrawableEdge(aNode, p1Node, "", (float) (1));
       edge.setColor(Color.blue);
       aNode.addOutEdge(edge);
       p1Node.addInEdge(edge);
       if (p2Node != null) {
         edge = new DefaultDrawableEdge(aNode, p2Node, "", (float) (1));
         edge.setColor(Color.blue);
         aNode.addOutEdge(edge);
         p2Node.addInEdge(edge);
      } else {
        System.out.println(p2+" doesn't have a node for itself!");
      }
    }

    protected void removeNode(Agent a) {
      Node aNode = (Node) nodeMap.get(a);

      // Safety check
      if (aNode==null)
         return;

      netList.remove(aNode);
      nodeMap.remove(a);

      Iterator i = aNode.getOutEdges().iterator();
      while (i.hasNext()) {
        Edge e = (Edge) i.next();
        Node bNode = e.getTo();
        Agent b = (Agent) nodeMap.get(bNode);
        // The partner might already have been removed.
        // In that case we do nothing since then the distance
        // has already been distracted from the sum.
        if (b != null) {
          // Everybody must be removed twice (since the relation is
          // symmetric, and the second time around the partner is not
          // going to be available.
          cumulDistance -= 2 * a.compareProps(b);
          numDistances -= 2;
          // We calculate it, independent of isAvgDistanceGraph,
          // because it would be impossible to produce the correct
          // value after a period with no calculations when isNetwork=true.
        }
        bNode.removeInEdge(e);
      }

      i = aNode.getInEdges().iterator();
      while (i.hasNext()) {
        Edge e = (Edge) i.next();
        Node bNode = e.getFrom();
        Agent b = (Agent) nodeMap.get(bNode);
        // The partner might already have been removed.
        // In that case we do nothing since then the distance
        // has already been distracted from the sum.
        if (b != null) {
          // Everybody must be removed twice (since the relation is
          // symmetric, and the second time around the partner is not
          // going to be available.
          cumulDistance -= 2 * a.compareProps(b);
          numDistances -= 2;
          // We calculate it, independent of isAvgDistanceGraph,
          // because it would be impossible to produce the correct
          // value after a period with no calculations when isNetwork=true.
        }
        bNode.removeOutEdge(e);
      }
    }

  public void addAgent(Agent a) {
    super.addAgent(a);

    if (isNetwork) {
      createNode(a);
    }
  }

    public Edge getEdgeToFrom(Node node, Node target)  {
      ArrayList edges = node.getInEdges();
      edges.addAll(node.getOutEdges());

      Iterator i = edges.iterator();
      while (i.hasNext()) {
        Edge e = (Edge) i.next();
        if ((e.getFrom() == target) || (e.getTo() == target))
          return e;
      }

      return null;
    }


//  public void removeAgent(Agent a) {
//    super.removeAgent(a);
//  }

  public void addPropToAll() {
    super.addPropToAll();

    if (isPropertyHistogram) {
      createPropertyHistogramItem(numProps-1);
    }

    if (isPropertyHistogram2) {
      propertyHistogram2.setUpperBound(numProps);
      propertyHistogram2.setNumBins(numProps);
    }
  }
////////////////////////////////////////////////////////////////////////////////
// Methods required for the parameters to show up on the 'parameter panel'.
////////////////////////////////////////////////////////////////////////////////

  public int getNumAgents() {
    return numAgents;
  }

  public void setNumAgents(int a) {
    numAgents = a;
  }

  public int getNumProps() {
    return numProps;
  }

  public void setNumProps(int a) {
    numProps = a;
  }

  public int getMinProp() {
    return minProp;
  }

  public void setMinProp(int a) {
    minProp = a;
  }

  public int getMaxProp() {
    return maxProp;
  }

  public void setMaxProp(int a) {
    maxProp = a;
  }

  public double getPEncounter() {
    return pEncounter;
  }

  public void setPEncounter(double d) {
    pEncounter = d;
  }

  public double getPCrossOver() {
    return pCrossOver;
  }

  public void setPCrossOver(double d) {
    pCrossOver = d;
  }

  public double getPMutation() {
    return pMutation;
  }

  public void setPMutation(double d) {
    pMutation = d;
  }

  public double getPNewSlot() {
    return pNewSlot;
  }

  public void setPNewSlot(double d) {
    pNewSlot = d;
  }

  public double getMappingLimit() {
    return mappingLimit;
  }

  public void setMappingLimit(double d) {
    mappingLimit = d;
  }

  public int getMappingMax() {
    return mappingMax;
  }

  public void setMappingMax(int i) {
    mappingMax = i;
  }

  public boolean getIsDeath() {
    return isDeath;
  }

  public void setIsDeath(boolean b) {
    isDeath = b;
  }

  public void setEConsumption(double d) {
    eConsumption = d;
  }

  public double getEConsumption() {
    return eConsumption;
  }

  public void setEInput(double d) {
    eInput = d;
  }

  public double getEInput() {
    return eInput;
  }

  public void setEDisc(double d) {
    eDisc = d;
  }

  public double getEDisc() {
    return eDisc;
  }

  public int getInitEnergy() {
    return initEnergy;
  }

  public void setInitEnergy(int a) {
    initEnergy = a;
  }

  public int getIncrEnergy() {
    return incrEnergy;
  }

  public void setIncrEnergy(int a) {
    incrEnergy = a;
  }

  public boolean getIsClustering() {
    return isClustering;
  }

  public void setIsClustering(boolean b) {
    isClustering = b;
  }

  public boolean getIsSpeciesNum() {
    return isSpeciesNum;
  }

  public void setIsSpeciesNum(boolean b) {
    isSpeciesNum = b;
  }

  public boolean getIsSpeciesDump() {
    return isSpeciesDump;
  }

  public void setIsSpeciesDump(boolean b) {
    isSpeciesDump = b;
  }

  public boolean getIsNetwork() {
    return isNetwork;
  }

  public void setIsNetwork(boolean b) {
    isNetwork = b;

    if (isRunning) {
      if (isNetwork) {
        netList = new ArrayList();
        nodeMap = new HashMap();

        cumulDistance = 0;
        numDistances = 0;

        // Here we take advantage of the symmetry in distance and
        // the handling of non-existent Node objects in createNode().
        Iterator i = agentList.iterator();
        while (i.hasNext()) {
          createNode((Agent) i.next());
        }

        if (netSurf == null) {
          createNetDisplay();
          if (netList.size() > 0) {
            netLayout.updateLayout();
          }

          if (isAvgDistanceGraph) {
            createAvgDistanceGraph();
            avgDistanceGraph.step();
          }
        }

      } else {
        netSurf.dispose();
        netSurf = null;
        netList = null;
        nodeMap = null;

        if (isAvgDistanceGraph) {
          avgDistanceGraph.dispose();
          avgDistanceGraph = null;
        }
      }
    }
  }

  public boolean getIsEnergyGraph() {
    return isEnergyGraph;
  }

  public void setIsEnergyGraph(boolean b) {
    isEnergyGraph = b;
  }

  public boolean getIsAgeHistogram() {
    return isAgeHistogram;
  }

  public void setIsAgeHistogram(boolean b) {
    isAgeHistogram = b;

    if (isRunning) {
      if (isAgeHistogram) {
        if (ageHistogram == null) {
          createAgeHistogram();
          ageHistogram.step();
        }
      } else {
        ageHistogram.dispose();
        ageHistogram = null;
      }
    }
  }

  public boolean getIsAvgDistanceGraph() {
    return isAvgDistanceGraph;
  }

  public void setIsAvgDistanceGraph(boolean b) {
    isAvgDistanceGraph = b;

    if (isRunning) {
      // We only make the dynamic showing/hiding happen if
      // the graph display is on.
      if (isNetwork) {
        if (isAvgDistanceGraph) {
          if (avgDistanceGraph == null) {
            createAvgDistanceGraph();
            avgDistanceGraph.step();
          }
        } else {
            avgDistanceGraph.dispose();
            avgDistanceGraph = null;
        }
      }
    }
  }

  public boolean getIsPropertyHistogram() {
    return isPropertyHistogram;
  }

  public void setIsPropertyHistogram(boolean b) {
    isPropertyHistogram = b;

    if (isRunning) {
      if (isPropertyHistogram) {
        if (propertyHistogram == null) {
          createPropertyHistogram();
          propertyHistogram.step();
        }
      } else {
          propertyHistogram.dispose();
          propertyHistogram = null;
      }
    }
  }

  public boolean getIsPropertyHistogram2() {
    return isPropertyHistogram2;
  }

  public void setIsPropertyHistogram2(boolean b) {
    isPropertyHistogram2 = b;

    if (isRunning) {
      if (isPropertyHistogram2) {
        if (propertyHistogram2 == null) {
          createPropertyHistogram2();
          propertyHistogram2.step();
        }
      } else {
          propertyHistogram2.dispose();
          propertyHistogram2 = null;
      }
    }
  }

////////////////////////////////////////////////////////////////////////////////
  /**
   * 'Magic' to start the model.
   *
   * @param args Parameters passed by the user. (Not used.)
   */
  public static void main(String[] args) {
    SimInit init = new SimInit();
    init.loadModel(new ModelGUI(), null, false);
  }

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

  /**
   * Inner class to represent species. Part of the 'magic; to calculate
   * clusters...
   */
  class Species {

    Agent agent;
    ArrayList popul;

    HashMap distances;

    public Species(Agent a) {
      this.agent = a;
      agentToSpecies.put(a, this);


      popul = new ArrayList();
      popul.add(a);

      distances = new HashMap();
    }

    public void calcInitDistances() {
      for (int i=0; i<agentList.size(); i++) {
        Agent other = (Agent) agentList.get(i);
        if (other != agent)
          distances.put(agentToSpecies.get(other), new Double(agent.compareProps(other)));
      }
    }

    public int getSize() {
      return popul.size();
    }

    public double getDistanceTo(Species other) {
      Double d = (Double) distances.get(other);
      return d.doubleValue();
    }

    public Species getClosest() {
      Species minIndex = null;
      double min = Double.MAX_VALUE;
      Iterator i = distances.entrySet().iterator();

      while (i.hasNext()) {
        Map.Entry e = (Map.Entry) i.next();
        Species other = (Species) e.getKey();
        Double d = (Double) e.getValue();

//System.out.println(d);
        if (d.doubleValue() < min) {
          min = d.doubleValue();
          minIndex = other;
        }
      }
//if (minIndex == null) {
//  System.out.println("DEBUG: "+distances.size()+", "+speciesList.size());
//}

      return minIndex;
    }

    public void mergeWith(Species other) {
      // Add it to the population
      popul.addAll(other.popul);

      // Delete other from distance map
      distances.remove(other);

      // Update distances
      Iterator i = distances.entrySet().iterator();
      while (i.hasNext()) {
        Map.Entry e = (Map.Entry) i.next();
        Species s = (Species) e.getKey();
        double d1 = ((Double) e.getValue()).doubleValue();
        double d2 = other.getDistanceTo(s);
        if (d2 > d1) {
          e.setValue(new Double(d2));
        }
      }
    }

    public void mergedAtoB(Species a, Species b) {
      double d1 = getDistanceTo(a);
      double d2 = getDistanceTo(b);

      if (d1 > d2) {
        distances.remove(b);
        distances.put(b, new Double(d1));
      }
      distances.remove(a);
    }

    public String toString() {
      Collections.sort(popul, new Comparator() {
                                public int compare(Object a, Object b) {
                                  if (a == b)
                                    return 0;

                                  if (((Agent) a).ID < ((Agent) b).ID)
                                    return -1;
                                  else
                                    return 1;
                                }
                              }
                      );
      return popul.toString();
    }
  }
}