package evotech.tools;

import uchicago.src.sim.analysis.*;
import uchicago.src.sim.analysis.plot.*;
import uchicago.src.sim.engine.SimModel;

import java.util.List;
import java.util.Vector;

/**
 * A dynamic bar chart. The OpenHistogram class allows the user to histogram data
 * generated by a collection of objects. The histogram range, as displayed, is
 * [lowerBound, maxValue] where the maxValue is calculated each time the graph
 * is stepped. The individual bin ranges except for the final bin are
 * [Math.floor(value), Math.ceil(value + interval)) where interval
 * is calculated by (maxValue - lowerBound) / numBins. The final bin is
 * inclusive.
 *
 * @author Nick Collier
 * @version $Revision: 1.5 $ $Date: 2002/12/18 14:44:01 $
 * @see HistogramStatistic
 * @see HistogramItem
 */

public class OpenHistogram2 extends OpenGraph {

  private OpenHistogramStat2 stats;
  private int dataset = 0;

  protected boolean xTickDrawn = false;

  /**
   * Constructs an OpenHistogram with the specified title, the specified number
   * of bins, and the specified lowerBound. A lowerBound higher than any
   * expected data values will cause that data not to be included in the
   * histogram.
   *
   * @param title the title of the histogram
   * @param numBins the number of bins
   * @param lowerBound the lower bound of the histogram
   */

  public OpenHistogram2(String title, int numBins, long lowerBound, long upperBound) {
    super(title);
    stats = new OpenHistogramStat2(numBins, lowerBound, upperBound);
    this.setBars(.5, .2);
    this.setXRange(0, numBins - 1);
  }

  /**
   * Constructs an OpenHistogram with the specified title, the specified number
   * of bins, and the specified lowerBound. A lowerBound higher than any
   * expected data values will cause that data not to be included in the
   * histogram. Model parameter necessary if you want to take snapshots or
   * movies of this graph.
   *
   * @param title the title of the histogram
   * @param numBins the number of bins
   * @param lowerBound the lower bound of the histogram
   * @param model the model associated with this graph
   */
  public OpenHistogram2(String title, int numBins,
                        long lowerBound, long upperBound, SimModel model) {
    this(title, numBins, lowerBound, upperBound);
    this.model = model;
  }

  /**
   * Creates a HistogramItem with specified name, list and BinDataSource to be
   * displayed by this Histogram.
   * The HistogramItem iterates over the specified list passing each Object
   * in the list as argument to the getBinValue method of the BinDataSource.
   * This getBinValue method returns a double. All these doubles
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * <b>Note:</b> This creates a Histogram that updates faster than
   * those created with createHistogramItem(String name, List list, String
   * methodName). Consequently, it should be preferred over the other method.
   *
   * @param name the name of the item
   * @param list the list of objects that provided the data for the item
   * @param source the BinDataSource used to get the data from the objects
   * in the list.
   */

  public void createHistogramItem(String name, List list, BinDataSource source) {
    stats.createHistogramItem(name, list, source);
    plot.addLegend(dataset++, name);
    this.setBars(1.0 / (double) dataset, 1.0 / (double) dataset);
  }

  /**
   * Creates a HistogramItem with specified name to be displayed by this
   * OpenHistogram.
   * The HistogramItem iterates over the specified list passing each Object
   * in the list as argument to the method with the specified name. This
   * method is a method of the specified object and returns a double. All these
   *  doubles
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of the item
   * @param list the list of objects that provided the data for the item
   * @param target the target of the name method
   * @param methodName the name of the method to call and pass each object
   * in the list to
   */
  public void createHistogramItem(String name, List list, Object target,
                                  String methodName) {
    BinDataSource bds = createBinDataSource(target, methodName);
    createHistogramItem(name, list, bds);
  }

  /**
   * Creates a HistogramItem with specified name to be displayed by this
   * OpenHistogram.
   * The HistogramItem iterates over the specified list passing each Object
   * in the list as argument to the method with the specified name. This
   * method is a method of the specified object and returns a double. All these
   *  doubles
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of the item
   * @param list the list of objects that provided the data for the item
   * @param target the target of the name method
   * @param methodName the name of the method to call and pass each object
   * in the list to
   * @param maxIntegerDigits the maximum number of digits before the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @param maxFractionDigits the maximum number of digits after the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   */
  public void createHistogramItem(String name, List list, Object target,
                                  String methodName, int maxIntegerDigits,
                                  int maxFractionDigits) {
    BinDataSource bds = createBinDataSource(target, methodName);
    createHistogramItem(name, list, bds, maxIntegerDigits, maxFractionDigits);
  }

  /**
   * Creates a HistogramItem with specified name, list and BinDataSource to be
   * displayed by this Histogram.
   * The HistogramItem iterates over the specified list passing each Object
   * in the list as argument to the getBinValue method of the BinDataSource.
   * This getBinValue method returns a double. All these doubles
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of the item
   * @param list the list of objects that provided the data for the item
   * @param source the BinDataSource used to get the data from the objects
   * in the list.
   * @param maxIntegerDigits the maximum number of digits before the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @param maxFractionDigits the maximum number of digits after the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   */

  public void createHistogramItem(String name, List list, BinDataSource source,
                                  int maxIntegerDigits, int maxFractionDigits) {
    stats.createHistogramItem(name, list, source, maxIntegerDigits,
                              maxFractionDigits);
    plot.addLegend(dataset++, name);
    this.setBars(1.0 / (double) dataset, 1.0 / (double) dataset);
  }


  /**
   * Creates a HistogramItem to be displayed by this OpenHistogram. The HistogramItem
   * iterates over the specified list calling the specified method on each
   * object in the list. This method must return a value that can be
   * cast into a double (i.e. float, int, double, short, etc.). These values
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * <b>Note:</b> This creates a Histogram that updates more slowly than
   * that created with createHistogramItem(String name, List list,
   * BinDataSource source). Consequently, this method should not be used
   * unless the Histogram will be updated infrequently.
   *
   * @param name the name of this item
   * @param list the list of object on which the specified method is called
   * @param listObjMethodName the name of the method to call on the objects. Should
   * return a Number value.
   *
   * @see HistogramItem
   */
  public void createHistogramItem(String name, List list,
                                  String listObjMethodName) {
    stats.createHistogramItem(name, list, listObjMethodName);
    plot.addLegend(dataset++, name);
    this.setBars(1.0 / (double) dataset, 1.0 / (double) dataset);
  }

  /**
   * Creates a HistogramItem to be displayed by this OpenHistogram. The HistogramItem
   * iterates over the specified list calling the specified method on each
   * object in the list. This method must return a value that can be
   * cast into a double (i.e. float, int, double, short, etc.). These values
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * <b>Note:</b> This creates a Histogram that updates more slowly than
   * that created with createHistogramItem(String name, List list,
   * BinDataSource source). Consequently, this method should not be used
   * unless the Histogram will be updated infrequently.
   *
   * @param name the name of this item
   * @param list the list of object on which the specified method is called
   * @param listObjMethodName the name of the method to call on the objects. Should
   * return a Number value.
   * @param maxIntegerDigits the maximum number of digits before the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @param maxFractionDigits the maximum number of digits after the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @see HistogramItem
   */
  public void createHistogramItem(String name, List list,
                                  String listObjMethodName,
                                  int maxIntegerDigits, int maxFractionDigits) {
    stats.createHistogramItem(name, list, listObjMethodName, maxIntegerDigits,
                              maxFractionDigits);
    plot.addLegend(dataset++, name);
    this.setBars(1.0 / (double) dataset, 1.0 / (double) dataset);
  }

  /**
   * Records data from the HistogramItems without updating the display
   */

  public void record() {
    stats.record();
  }

  /**
   * Records any new data and updates the displayed graph.
   */
  public void step() {
    stats.record();
    updateGraph();
  }

  /**
   * Updates the graph.
   */
  public void updateGraph() {

    plot.clearPoints();

    if (!xTickDrawn) {
      String[] labels = stats.getPointLabels();
      for (int i = 0; i < labels.length; i++) {
        this.updateXTick(i, labels[i], i);
      }
      xTickDrawn = true;
    }

    Vector table = stats.getDataTable();
    Vector xVals = (Vector) table.get(0);
    for (int i = 0; i < xVals.size(); i++) {
      double xVal = ((Double) xVals.get(i)).doubleValue();

      for (int j = 1; j < table.size(); j++) {
        Vector yVals = (Vector) table.get(j);
        double yVal = ((Double) yVals.get(i)).doubleValue();

        plot.addPoint(j - 1, xVal, yVal, false);
      }
    }

    //plot.repaint();
    plot.fillPlot();
  }

  public void setUpperBound(int upperBound) {
    this.stats.setUpperBound(upperBound);
  }

  public void setNumBins(int numBins) {
    this.stats.setNumBins(numBins);
    this.xTickDrawn = false;
  }
}
