package evotech.tools;

import uchicago.src.sim.analysis.*;

import java.util.*;

/**
 * The Statistics class for histogram data. Operates as the backend for a
 * Histogram, manipulating the HistogramItems. Can be created without a
 * Histogram class to generate histogram data without displaying such.
 *
 * @author Nick Collier
 * @version $Revision: 1.4 $ $Date: 2002/11/04 16:58:44 $
 * @see Histogram
 * @see HistogramItem
 */

public class OpenHistogramStat2 extends OpenStats {

  private Vector items = new Vector();
  private int numBins = 0;
  private long lowerBound;
  private long upperBound;
  private double interval;

  protected boolean isLabelsSet = false;

  /**
   * Constructs a HistogramStatistic with the specified number of bins and the
   * specified lower bound.
   *
   * @param numBins the number of bins
   * @param lowerBound the lower bound
   */
  public OpenHistogramStat2(int numBins, long lowerBound, long upperBound) {
    super(null);
    this.numBins = numBins;
    this.lowerBound = lowerBound;
    this.upperBound = upperBound;
    for (int i = 0; i < numBins; i++) {
      data.addX(i);
    }
  }


  /**
   * Creates a HistogramItem. The HistogramItem
   * iterates over the specified list calling the specified method on each
   * object in the list. This method must return a value that can be
   * cast into a double (i.e. float, int, double, short, etc.). These values
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of this item
   * @param list the list of object on which the specified method is called
   * @param methodName the name of the method to call on the objects. Should
   * return a Number value.
   * @see HistogramItem
   */

  public void createHistogramItem(String name, List list, String methodName) {
    HistogramItem2 item = new HistogramItem2(list, methodName, numBins, lowerBound, upperBound);
    data.addSeries(name);
    items.addElement(item);
  }

  /**
   * Creates a HistogramItem. The HistogramItem
   * iterates over the specified list calling the specified method on each
   * object in the list. This method must return a value that can be
   * cast into a double (i.e. float, int, double, short, etc.). These values
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of this item
   * @param list the list of object on which the specified method is called
   * @param methodName the name of the method to call on the objects. Should
   * return a Number value.
   * @param maxIntegerDigits the maximum number of digits before the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @param maxFractionDigits the maximum number of digits after the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @see HistogramItem
   */

  public void createHistogramItem(String name, List list, String methodName,
                                  int maxIntegerDigits, int maxFractionDigits) {
    HistogramItem2 item = new HistogramItem2(list, methodName, numBins,
                                           lowerBound, upperBound,
                                           maxIntegerDigits, maxFractionDigits);
    data.addSeries(name);
    items.addElement(item);
  }

  /**
   * Creates a HistogramItem with specified name, list and BinDataSource.
   * The HistogramItem iterates over the specified list passing each Object
   * in the list as argument to the getBinValue method of the BinDataSource.
   * This getBinValue method returns a double. All these doubles
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of the item
   * @param list the list of objects that provided the data for the item
   * @param source the BinDataSource used to get the data from the objects
   * in the list.
   */
  public void createHistogramItem(String name, List list, BinDataSource source) {
    HistogramItem2 item = new HistogramItem2(list, source, numBins,
                                             lowerBound, upperBound);
    data.addSeries(name);
    items.addElement(item);
  }

  /**
   * Creates a HistogramItem with specified name, list and BinDataSource.
   * The HistogramItem iterates over the specified list passing each Object
   * in the list as argument to the getBinValue method of the BinDataSource.
   * This getBinValue method returns a double. All these doubles
   * are then distributed across the bins according to the number of bins,
   * the lower bound, and the maximum value. For example, given 2 bins,
   * a lower bound of 0 and a maximum value of 4. The first bin will contain
   * all the values from 0 up to but not including 2, and the final bin will
   * contain all the values from 2 up to <em>and</em> including 4. The displayed
   * bin value (i.e. the height of the bar in the chart) is the number of values
   * that fall within this bin.<p>
   *
   * @param name the name of the item
   * @param list the list of objects that provided the data for the item
   * @param source the BinDataSource used to get the data from the objects
   * in the list.
   * @param maxIntegerDigits the maximum number of digits before the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   * @param maxFractionDigits the maximum number of digits after the
   * decimal point in the bin labels. A value of -1 will record all the digits.
   */
  public void createHistogramItem(String name, List list, BinDataSource source,
                                  int maxIntegerDigits, int maxFractionDigits) {
    HistogramItem2 item = new HistogramItem2(list, source, numBins,
                                             lowerBound, upperBound,
                                             maxIntegerDigits, maxFractionDigits);
    data.addSeries(name);
    items.addElement(item);
  }

  /**
   * Calculates a new histogram over the data supplied by a HistogramItem
   */
  public void record() {
    Vector yVals;
    for (int i = 0; i < items.size(); i++) {
      HistogramItem2 item = (HistogramItem2) items.elementAt(i);
      yVals = item.getBinValues();
      data.setYValues(yVals, i);
      if (!isLabelsSet) {
        data.setPointLabels(item.getBinLabels());
        isLabelsSet = true;
      }
    }
  }

  /**
   * Sets the histogram labels (the x - axis legend).
   */
  public void setPointLabels(String[] labels) {
    data.setPointLabels(labels);
  }

  /**
   * Write the histogram data to a file. This method is not yet implemented.
   */
  public void writeToFile() {
    throw new UnsupportedOperationException("This method is not yet implemented");

  }

  public void setUpperBound(int upperBound) {
    this.upperBound = upperBound;
    Iterator i = this.items.iterator();
    while (i.hasNext()) {
      HistogramItem2 h2 = (HistogramItem2) i.next();
      h2.setUpperBound(upperBound);
    }
  }

  public void setNumBins(int numBins) {
    this.numBins = numBins;
    data = new OpenArrayData();
    for (int j=0; j<numBins; j++) {
      data.addX(j);
    }

    Iterator i = this.items.iterator();
    while (i.hasNext()) {
      HistogramItem2 h2 = (HistogramItem2) i.next();
      h2.setNumBins(numBins);
      // Hack by Gulya
      // Needed this way, because getSeriesName() is not properly implemented
      // in OpenDataArray.
      data.addSeries("#");
    }
    this.isLabelsSet = false;
  }
}
