/*
 * cache.c - object cache procedures for KOTETU
 *      by k-chinen@is.aist-nara.ac.jp, 1999
 *
 * $Id$
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <pthread.h>
#include <errno.h>

#include "range.h"
#include "msg.h"
#include "cache.h"
/* why ?
#include "http.h"
*/


off_t cachetotal        = -1;
off_t lfu_area_size     = -1;
off_t htrie_area_size    = -1;

/* Hashed tries for small object */
unsigned char *_htrie_slot = NULL;
int nline        = -1;
int nblock      = -1;
int linesize    = -1;
int datasize    = -1;


/* LFU for large object */
unsigned char *_lfu_slot = NULL;
int lnslot      = -1;
int slotsize    = -1;
int ldatasize   = -1;
pthread_mutex_t lfu_mutex = PTHREAD_MUTEX_INITIALIZER;

/*
    rq_size:    require size
    rq_m:       require number of data for LFU
    rq_jsize:   require each data size for LFU
    rq_bsize:   require each block size for Hashed Tries
    rq_dsize:   require each data size in line for Hashed Tries
*/
int
calc_param(
    off_t rq_size, int rq_m, off_t rq_jsize, off_t rq_bsize, int rq_dsize)
{
    size_t psize;
    size_t lcsize;

    fprintf(stderr, "calc_param:\n");
    fprintf(stderr, "  requirements:\n");
    fprintf(stderr, "   total size %d bytes(%ldM+%ld)\n",
        (long)rq_size, (long)dM(rq_size), (long)mM(rq_size));
    fprintf(stderr, "   <LFU>   %d x %ld bytes(%ldM+%ld)\n",
        rq_m, (long)rq_jsize, (long)dM(rq_jsize), (long)mM(rq_jsize));
    fprintf(stderr, "   <Hashed Tries> block %ld bytes(%ldK+%ld), line %ld bytes(%ldK+%ld)\n",
        (long)rq_bsize, (long)dK(rq_bsize), (long)mK(rq_bsize),
        (long)rq_dsize, (long)dK(rq_dsize), (long)mK(rq_dsize));

    fprintf(stderr, "\n");

    psize = getpagesize();
    fprintf(stderr, "page size of this system is %d bytes(%dK+%d)\n",
        psize, dK(psize), mK(psize));

    fprintf(stderr, "\n");

    slotsize = ((rq_jsize+psize-1)/psize)*psize;
    lnslot = rq_m;
    lfu_area_size = slotsize*lnslot;
    fprintf(stderr, "LFU: solve %ld bytes(%ldM+%ld) x %d slots\n",
        (long)slotsize, (long)dM(slotsize), (long)mM(slotsize), lnslot);
    fprintf(stderr, "   area size %d bytes(%ldM+%ld)\n",
        (long)lfu_area_size, (long)dM(lfu_area_size), (long)mM(lfu_area_size));

    if(rq_size<lfu_area_size) {
        fprintf(stderr, "LFU area is over running on cache area\n");
        exit(87);
    }


    fprintf(stderr, "\n");

    lcsize = sizeof(line_cntl_rec);
    datasize = ((rq_dsize+lcsize+psize-1)/psize)*psize - lcsize;
    linesize = lcsize+datasize;
    ldatasize = slotsize - lcsize;

    fprintf(stderr, "Hashed Tries: size of line control block is %d\n",
        lcsize);
    fprintf(stderr, "    then, size of data is %d\n", datasize);
    fprintf(stderr, "    size of line is %d (%dK+%d)\n\n",
        linesize, dK(linesize), mK(linesize));

    htrie_area_size = rq_size - lfu_area_size;


    nline = rq_bsize/linesize;
    nblock = htrie_area_size/(nline*linesize);

    fprintf(stderr, "    the number of lines is %ld (included 1 block control line)\n", (long)nline);
    fprintf(stderr, "    the number of blocks is %ld\n", (long)nblock);

    fprintf(stderr, "\n");

#if 0
    cachetotal = lfu_area_size + nblock*nline*linesize;
#endif
    cachetotal = htrie_area_size + lfu_area_size;

    fprintf(stderr, "total size of cache %ld (%ldM+%ld)\n\n",
        (long)cachetotal, (long)(cachetotal/(1024*1024)),
        (long)(cachetotal%(1024*1024)));


    if(nblock==0) {
        return -1;
    }
    return 0;
}


int
cache_alloc_area(off_t rq_size, int rq_m, off_t rq_jsize, off_t rq_bsize, int rq_dsize)
{
    int l;
    int w;
#if 0
    line_rec *lr;
#endif
    char *lr;
    block_cntl_rec *bcr;
    line_cntl_rec  *lcr;

    calc_param(rq_size, rq_m, rq_jsize, rq_bsize, rq_dsize);

    if(nline<=1) {
        exit(3);
    }

#if 0
    _htrie_slot = (unsigned char*)malloc((size_t)htrie_area_size);
    if(_htrie_slot == NULL) {
        exit(4);
    }
#else
 {
    unsigned char *tmp;

    /* for debug people, cut right side */
    tmp = (unsigned char*)malloc((size_t)(htrie_area_size + 1024));
    if(tmp == NULL) {
	Error("not enought memory for cache object area (size %ul)",
		(unsigned long)htrie_area_size);
        exit(4);
    }
    tmp = (unsigned char*)((((long long)tmp >> 10)+1) << 10);

    _htrie_slot= (unsigned char*) tmp;
 }
#endif
    Log("; address of _htrie_slot = %p (size %d)",
        _htrie_slot, htrie_area_size);

    for(l=0;l<nblock;l++) {
        lr = (char*)&_htrie_slot[nline*linesize*l];

        bcr = (block_cntl_rec*)&lr[0];
        bcr->type = TYPE_BLOCK_CONTROL;
        if(pthread_mutex_init(&bcr->mutex, NULL)<0) {
            fprintf(stderr, "pthread_mutex_init() fail %d\n", errno);
        }

        lr = (char*)bcr + linesize;

        for(w=1;w<nline;w++) {
            lcr = (line_cntl_rec*) lr;

            lcr->type     = TYPE_EMPTY_LINE;
            lcr->lhash    = -1;
            lcr->name[0]  = '\0';
            lcr->hot      = 0;
            lcr->count    = 0;
            lcr->cracked  = 0;
            lcr->size     = datasize;
            lcr->length   = -1;
            lcr->lmd      = -1;

            ((line_rec*)lcr)->body  =
                (unsigned char*) lcr + sizeof(line_cntl_rec);
#if 1
            memset(((line_rec*)lcr)->body, 0, datasize);
#endif

            lr += linesize;
        }

#if 0
        lr = (char*)bcr + linesize;
        for(w=1;w<nline;w++) {
            lcr = (line_cntl_rec*) lr;

            Trace("%4d: lr %p : cntl %p, body %p\n",
                w, lr, lcr, &((line_rec*)lr->body) );

            lr += linesize;
        }

        /*
        exit(1);
        */
#endif
    }

    _lfu_slot = (unsigned char*)malloc((size_t)lfu_area_size);
    if(_lfu_slot == NULL) {
        exit(5);
    }


    for(w=0;w<lnslot;w++) {
        lr = (char*)&_lfu_slot[w*slotsize];
        lcr = (line_cntl_rec*) lr;
        lcr->type       = TYPE_EMPTY_LINE;
        lcr->lhash      = -1;
        lcr->name[0]    = '\0';
        lcr->hot        = 0;
        lcr->count      = 0;
        lcr->size       = ldatasize;
        lcr->length     = -1;

        ((line_rec*)lcr)->body  =
            (unsigned char*) lcr + sizeof(line_cntl_rec);
    }


    return 0;
}


#ifdef USE_SIMPLE_FIRST_HASH

#ifdef USE_VERY_SIMPLE_FIRST_HASH

int
first_hash(unsigned char *b)
{
    return 0;
}

#else

int
first_hash(unsigned char *b)
{
    int l;
    register unsigned int sum, c;
    register unsigned char *p;

    sum = 0U;

    p = b;
    c = 0;
    while(*p) {
        sum += *p++;
        c++;
    }

    return c % nblock;
}

#endif /* USE_VERY_SIMPLE_HASH */

#else

/* used for index */
static int prime_box[] = {23,19,13,31,11,67,7,71,5,3,29};

int
first_hash(unsigned char *b)
{
    int l;
    register int sum, c;
    register unsigned char *p, *r, *s;

    sum = 0;

    c = 1;
    p = b;
    while(*p) {
        sum += *p - ' ' + c;
        if(*p>='0' && *p<='9') {
            /* sum += (*p - '0' + 3) * 3; */
            sum += prime_box[*p - '0'] * c;
        }
        p++;
        c++;
    }

    if(sum<0)
        sum = -sum;

    return sum % nblock;
}

#endif


/* used for identify */
int
second_hash(unsigned char *b)
{
    register int sum;
    register unsigned char *p;

    sum = 0;

    p = b;
    while(*p) {
        sum = sum<<1;
        sum += *p;
        p++;
    }

    p--;
    while(p>b) {
        sum = sum<<1;
        sum += *p;
        p--;
    }

    if(sum<0)
        sum = -sum;

    return sum % 4099;
}

#ifndef NO_MEMORY_CHECK
    /*
     * this routine would be defined in cache.h
     */
    /*
    return (void*)&_htrie_slot[nline*linesize*fh];
    */
#else
void*
cache_block_pos(int fh)
{
    int pos;

    if(fh<0 || fh>=nblock) {
        return (void*)NULL;
    }
    pos = nline*linesize*fh;
    if(pos>=htrie_area_size) {
        return (void*)NULL;
    }
    else {
        return (void*)&_htrie_slot[pos];
    }
}
#endif



static pthread_mutex_t cache_show_lock = PTHREAD_MUTEX_INITIALIZER;

int
cache_showblock(int fh, int flag_lock)
{
    int ret;
    int chk;
    char *bpos;
    block_cntl_rec *bcr;
    int w;
    line_cntl_rec *lcr;
    char *clpos;
    char *lpos, *fpos;
    int rcode;
    int sum;
    char left[K(1)],right[K(8)];

    ret = 0;

    bpos = (char*) cache_block_pos(fh);
    bcr = (block_cntl_rec*)bpos;


    if(flag_lock) {
        Trace("cache_showblock: fh %d, bpos %p, bcr %p, datasize %d\n",
            fh, bpos, bcr, datasize);

        chk = MUTEX_TRYLOCK(&bcr->mutex);
        if(chk) {
            if(chk==EBUSY) {
                Error("cache_showblock: block #%d is busy", fh);
                ret = -1;
            }
            else {
                Error("cache_showblock: fail locking block #%d (%d)", fh, chk);
                ret = -2;
            }
            goto out;
        }
    }

    MUTEX_LOCK(&cache_show_lock);

    if(!flag_lock) {
        Trace("cache_showblock: fh %d, bpos %p, bcr %p, datasize %d\n",
            fh, bpos, bcr, datasize);
    }

    Trace("#%3s %c %c %5s %1s %5s %7s %4s '%s'\n",
        "no", 'T', 'C', "owner", "c", "len.", "sum", "hash", "URL");

    clpos = bpos  + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;
        if(lcr->type==TYPE_SLAVE_LINE) {
            /* nothing */
        }
        else {
            sum = 0;
        }
        if(lcr->length>0) {
            sum += lcr->length;
        }

        sprintf(left, " %3d %c%c%c %5d %d %5d %7d ", w,
            lcr->type==TYPE_EMPTY_LINE ? 'E' :
                (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                    ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
            lcr->hot ? '*' : '.',
            lcr->cracked ? 'C' : '.',
            lcr->owner,
            lcr->count,
            lcr->length, sum
          );

        if(lcr->type==TYPE_NORMAL_LINE) {
            sprintf(right, "%4d '%s'", lcr->lhash, lcr->name);
        }
        else
        if(lcr->type==TYPE_SLAVE_LINE) {
            sprintf(right, "same");
        }
        else
        if(lcr->type==TYPE_EMPTY_LINE) {
            sprintf(right, "-");
        }
        else {
        }
        Trace("%s %s\n", left, right);

        clpos += linesize;
    }

    MUTEX_UNLOCK(&cache_show_lock);


    if(flag_lock) {
        chk = MUTEX_UNLOCK(&bcr->mutex);
    }

out:
    return ret;
}



int
cache_showblockaddr(int fh, int flag_lock)
{
    int ret;
    int chk;
    char *bpos;
    block_cntl_rec *bcr;
    int w;
    line_cntl_rec *lcr, *h_lcr;
    char *clpos;
    char *lpos, *fpos;
    int rcode;
    int sum;
    char left[K(1)],right[K(8)];

    line_cntl_rec *mark[K(2)];
    int m, mark_count;
    int skip;

    ret = 0;

    bpos = (char*) cache_block_pos(fh);
    bcr = (block_cntl_rec*)bpos;


    if(flag_lock) {
        Trace("cache_showblockaddr: fh %d, bpos %p, bcr %p, datasize %d\n",
            fh, bpos, bcr, datasize);

        chk = MUTEX_TRYLOCK(&bcr->mutex);
        if(chk) {
            if(chk==EBUSY) {
                Error("cache_showblockaddr: block #%d is busy", fh);
                ret = -1;
            }
            else {
                Error("cache_showblock: fail locking block #%d (%d)", fh, chk);
                ret = -2;
            }
            goto out;
        }
    }

    MUTEX_LOCK(&cache_show_lock);

    if(!flag_lock) {
        clpos = bpos + linesize;
        Trace("block# %d, bpos %p, first %p, datasize %d\n",
            fh, bpos, clpos, datasize);
    }

    Trace("#%3s %8s %8s %c %c %5s %2s %7s %7s %4s '%s'\n",
        "no", "addr", "next",
        'T', 'C', "owner", "cn", "len.", "sum", "hash", "URL");


#if 0
    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;
        Trace(" %3d %8x %8x %c%c%c\n",
            w, lcr, lcr->next,
            lcr->type==TYPE_EMPTY_LINE ? 'E' :
                (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                    ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
            lcr->hot ? '*' : '.',
            lcr->cracked ? 'C' : '.');
        clpos += linesize;
    }
#endif

    mark_count = 0;

    sum = -1;
    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;

        skip = 0;
        if(lcr) {
            for(m=0;m<mark_count;m++) {
                if(mark[m] == lcr) {
                    skip = 1;
                    break;
                }
            }
        }
        if(!skip && lcr) {
            if(lcr->type==TYPE_NORMAL_LINE) {
                mark[mark_count++] = lcr;

                if(!lcr->cracked && lcr->totallength<lcr->length) {
                    Trace(" = = = strage size\n");
                }
                if(!lcr->cracked && lcr->next==NULL
                  && lcr->totallength!=lcr->length) {
                    Trace(" = = = strage size\n");
                }

                Trace(" %3d %8x %8x %c%c%c %5d %2d %7d* %6d %4d %s\n",
                    w, lcr, lcr->next,
                    lcr->type==TYPE_EMPTY_LINE ? 'E' :
                        (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                            ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                    lcr->hot ? '*' : '.',
                    lcr->cracked ? 'C' : '.',
                    lcr->owner,
                    lcr->count,
                    lcr->totallength,
                    lcr->length,
                    lcr->lhash, lcr->name);

                h_lcr = lcr;
                sum = lcr->length;
                lcr = lcr->next;

                while(lcr) {
                    mark[mark_count++] = lcr;

                    sum += lcr->length;
                    Trace(" %3d %8x %8x %c%c%c %5d %2d %7d %7d same\n",
                        w, lcr, lcr->next,
                        lcr->type==TYPE_EMPTY_LINE ? 'E' :
                            (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                                ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                        lcr->hot ? '*' : '.',
                        lcr->cracked ? 'C' : '.',
                        lcr->owner,
                        lcr->count,
                        lcr->length, sum);

                    if(lcr->type!=TYPE_SLAVE_LINE) {
                        break;
                    }
                    if(!h_lcr->cracked &&
                      lcr->next == NULL && h_lcr->totallength != sum) {
                        Trace("    size miss match\n");
                    }

                    lcr = lcr->next;
                }

            }
            else
            if(lcr->type==TYPE_EMPTY_LINE) {
                mark[mark_count++] = lcr;

                Trace(" %3d %8x %8x %c%c%c %5d %2d -\n",
                    w, lcr, lcr->next,
                    lcr->type==TYPE_EMPTY_LINE ? 'E' :
                        (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                            ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                    lcr->hot ? '*' : '.',
                    lcr->cracked ? 'C' : '.',
                    lcr->owner,
                    lcr->count);
            }
#if 0
            else {
                Trace("ignore\n");
            }
#endif
        }
        clpos += linesize;
    }

    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;

        skip = 0;
        for(m=0;m<mark_count;m++) {
            if(mark[m] == lcr) {
                skip = 1;
                break;
            }
        }
        if(!skip) {
            mark[mark_count++] = lcr;

            sprintf(left, "*%3d %8x %8x %c%c%c ",
                w, lcr, lcr->next,
                lcr->type==TYPE_EMPTY_LINE ? 'E' :
                    (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                        ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                lcr->hot ? '*' : '.',
                lcr->cracked ? 'C' : '.');

            if(lcr->type==TYPE_NORMAL_LINE) {
                sprintf(right, "%5d %d %5d %7d %4d %s",
                    lcr->owner,
                    lcr->count,
                    lcr->length, sum,
                    lcr->lhash, lcr->name);
            }
            else
            if(lcr->type==TYPE_SLAVE_LINE) {
                sprintf(right, "%5d %d %5d %7d same",
                    lcr->owner,
                    lcr->count,
                    lcr->length, sum,
                    lcr->lhash, lcr->name);
            }
            else
            if(lcr->type==TYPE_EMPTY_LINE) {
                sprintf(right, "%5d %d -",
                    lcr->owner,
                    lcr->count);
            }
            else {
            }
            Trace("%s %s\n", left, right);

        }
        clpos += linesize;
    }




    MUTEX_UNLOCK(&cache_show_lock);


    if(flag_lock) {
        chk = MUTEX_UNLOCK(&bcr->mutex);
    }

out:
    return ret;
}

int
cache_showblockaddr_fd(int fh, int flag_lock, int ofd)
{
    int ret;
    int chk;
    char *bpos;
    block_cntl_rec *bcr;
    int w;
    line_cntl_rec *lcr, *h_lcr;
    char *clpos;
    char *lpos, *fpos;
    int rcode;
    int sum;
    char left[K(1)],right[K(8)];

    line_cntl_rec *mark[K(2)];
    int m, mark_count;
    int skip;

    ret = 0;

    bpos = (char*) cache_block_pos(fh);
    bcr = (block_cntl_rec*)bpos;


    if(flag_lock) {
        fdprintf(ofd, "cache_showblockaddr: fh %d, bpos %p, bcr %p, datasize %d\n",
            fh, bpos, bcr, datasize);

        chk = MUTEX_TRYLOCK(&bcr->mutex);
        if(chk) {
            if(chk==EBUSY) {
                Error("cache_showblockaddr: block #%d is busy", fh);
                ret = -1;
            }
            else {
                Error("cache_showblock: fail locking block #%d (%d)", fh, chk);
                ret = -2;
            }
            goto out;
        }
    }

    MUTEX_LOCK(&cache_show_lock);

    if(!flag_lock) {
        clpos = bpos + linesize;
        fdprintf(ofd, "block# %d, bpos %p, first %p, datasize %d\n",
            fh, bpos, clpos, datasize);
    }

    fdprintf(ofd, "#%3s %8s %8s %c %c %5s %2s %7s %7s %4s '%s'\n",
        "no", "addr", "next",
        'T', 'C', "owner", "cn", "len.", "sum", "hash", "URL");


#if 0
    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;
        fdprintf(ofd, " %3d %8x %8x %c%c%c\n",
            w, lcr, lcr->next,
            lcr->type==TYPE_EMPTY_LINE ? 'E' :
                (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                    ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
            lcr->hot ? '*' : '.',
            lcr->cracked ? 'C' : '.');
        clpos += linesize;
    }
#endif

    mark_count = 0;

    sum = -1;
    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;

        skip = 0;
        if(lcr) {
            for(m=0;m<mark_count;m++) {
                if(mark[m] == lcr) {
                    skip = 1;
                    break;
                }
            }
        }
        if(!skip && lcr) {
            if(lcr->type==TYPE_NORMAL_LINE) {
                mark[mark_count++] = lcr;

                if(!lcr->cracked && lcr->totallength<lcr->length) {
                    fdprintf(ofd, " = = = strage size\n");
                }
                if(!lcr->cracked && lcr->next==NULL
                  && lcr->totallength!=lcr->length) {
                    fdprintf(ofd, " = = = strage size\n");
                }

                fdprintf(ofd, " %3d %8x %8x %c%c%c %5d %2d %7d* %6d %4d %s\n",
                    w, lcr, lcr->next,
                    lcr->type==TYPE_EMPTY_LINE ? 'E' :
                        (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                            ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                    lcr->hot ? '*' : '.',
                    lcr->cracked ? 'C' : '.',
                    lcr->owner,
                    lcr->count,
                    lcr->totallength,
                    lcr->length,
                    lcr->lhash, lcr->name);

                h_lcr = lcr;
                sum = lcr->length;
                lcr = lcr->next;

                while(lcr) {
                    mark[mark_count++] = lcr;

                    sum += lcr->length;
                    fdprintf(ofd, " %3d %8x %8x %c%c%c %5d %2d %7d %7d same\n",
                        w, lcr, lcr->next,
                        lcr->type==TYPE_EMPTY_LINE ? 'E' :
                            (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                                ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                        lcr->hot ? '*' : '.',
                        lcr->cracked ? 'C' : '.',
                        lcr->owner,
                        lcr->count,
                        lcr->length, sum);

                    if(lcr->type!=TYPE_SLAVE_LINE) {
                        break;
                    }
                    if(!h_lcr->cracked &&
                      lcr->next == NULL && h_lcr->totallength != sum) {
                        fdprintf(ofd, "    size miss match\n");
                    }

                    lcr = lcr->next;
                }

            }
            else
            if(lcr->type==TYPE_EMPTY_LINE) {
                mark[mark_count++] = lcr;

                fdprintf(ofd, " %3d %8x %8x %c%c%c %5d %2d -\n",
                    w, lcr, lcr->next,
                    lcr->type==TYPE_EMPTY_LINE ? 'E' :
                        (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                            ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                    lcr->hot ? '*' : '.',
                    lcr->cracked ? 'C' : '.',
                    lcr->owner,
                    lcr->count);
            }
#if 0
            else {
                fdprintf(ofd, "ignore\n");
            }
#endif
        }
        clpos += linesize;
    }

    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;

        skip = 0;
        for(m=0;m<mark_count;m++) {
            if(mark[m] == lcr) {
                skip = 1;
                break;
            }
        }
        if(!skip) {
            mark[mark_count++] = lcr;

            sprintf(left, "*%3d %8x %8x %c%c%c ",
                w, lcr, lcr->next,
                lcr->type==TYPE_EMPTY_LINE ? 'E' :
                    (lcr->type==TYPE_NORMAL_LINE ? 'N' :
                        ( lcr->type==TYPE_SLAVE_LINE ? 'S' : '?')),
                lcr->hot ? '*' : '.',
                lcr->cracked ? 'C' : '.');

            if(lcr->type==TYPE_NORMAL_LINE) {
                sprintf(right, "%5d %d %5d %7d %4d %s",
                    lcr->owner,
                    lcr->count,
                    lcr->length, sum,
                    lcr->lhash, lcr->name);
            }
            else
            if(lcr->type==TYPE_SLAVE_LINE) {
                sprintf(right, "%5d %d %5d %7d same",
                    lcr->owner,
                    lcr->count,
                    lcr->length, sum,
                    lcr->lhash, lcr->name);
            }
            else
            if(lcr->type==TYPE_EMPTY_LINE) {
                sprintf(right, "%5d %d -",
                    lcr->owner,
                    lcr->count);
            }
            else {
            }
            fdprintf(ofd, "%s %s\n", left, right);

        }
        clpos += linesize;
    }




    MUTEX_UNLOCK(&cache_show_lock);


    if(flag_lock) {
        chk = MUTEX_UNLOCK(&bcr->mutex);
    }

out:
    return ret;
}

int
cache_list_entry(FILE *ofp)
{
    int l;
    int w;
    char *lp;
    line_rec *lr;
    block_cntl_rec *bcr;
    line_cntl_rec  *lcr;
    int u;
    int chk;

    if(_htrie_slot == NULL) {
        fprintf(ofp,"no Hashed Tries slots\n");
        return -1;
    }
    if(_lfu_slot == NULL) {
        fprintf(ofp,"no LFU slots\n");
        return -1;
    }


    fprintf(ofp,"\n");

    fprintf(ofp, "Hashed Tries ===\n");
    fprintf(ofp, "nline %d, linesize %d (%ldK+%ld), datasize %d (%ldK+%ld)\n",
        nline, linesize, dK(linesize), mK(linesize),
        datasize, dK(datasize), mK(datasize));
    fprintf(ofp, "\n");

    for(l=0;l<nblock;l++) {
#if 0
        lr = (line_rec*)&_htrie_slot[nline*linesize*l];
        bcr = (block_cntl_rec*)&lr[0];
#endif
        lr = (line_rec*) cache_block_pos(l);
        bcr = (block_cntl_rec*)&lr[0];


        /* check lockable or not */
        u = 0;
        chk = MUTEX_TRYLOCK(&bcr->mutex);
        if(chk) {
            if(chk==EBUSY) {
                u = 1;
            }
        }
        else {
            MUTEX_UNLOCK(&bcr->mutex);
        }

        fprintf(ofp,"block #%d %p: %s : control: type %d\n",
            l, bcr, u==1 ? "USE" : "-", bcr->type);

        lp = (char*)bcr + linesize;
        for(w=1;w<nline;w++) {
            lr = (line_rec*) lp;
            fprintf(ofp,"    #%4d: %p : %c %c '%s'(%d) %d %c %d/%d\n",
                w, lr,
                lr->cntl.type==TYPE_NORMAL_LINE ? 'N' : '?',
                lr->cntl.hot>0 ? 'H' : 'c',
                lr->cntl.name, lr->cntl.lhash,
                lr->cntl.count,
                lr->cntl.cracked ? 'C' : '-',
                lr->cntl.length, lr->cntl.size);
            lp += linesize;
        }
    }

    fprintf(ofp,"LFU ===\n");
    for(w=0;w<lnslot;w++) {
        lr = (line_rec*)&_lfu_slot[w*slotsize];
        fprintf(ofp,"    #%4d: %p : %c %c '%s'(%d) %d %c %d/%d\n",
            w, lr,
            lr->cntl.type==TYPE_NORMAL_LINE ? 'N' : '?',
            lr->cntl.hot>0 ? 'H' : 'c',
            lr->cntl.name, lr->cntl.lhash,
            lr->cntl.count,
            lr->cntl.cracked ? 'C' : '-',
            lr->cntl.length, lr->cntl.size);
    }

    return 0;
}

int
cache_list_entry_fd(int ofd)
{
    int l;
    int w;
    char *lp;
    line_rec *lr;
    block_cntl_rec *bcr;
    line_cntl_rec  *lcr;
    int u;
    int chk;
    int obj_total_line, obj_total_slot;
    off_t obj_total_areasize, obj_total_usesize;
    int ncount, scount, ecount, zcount;


    if(_htrie_slot == NULL) {
        fdprintf(ofd,"no Hashed Tries slots\n");
        return -1;
    }
    if(_lfu_slot == NULL) {
        fdprintf(ofd,"no LFU slots\n");
        return -1;
    }


    fdprintf(ofd,"\n");

    obj_total_line = 0;
    obj_total_slot  = 0;
    obj_total_areasize = (off_t) 0;
    obj_total_usesize  = (off_t) 0;
    ncount = scount = ecount = zcount = 0;

    fdprintf(ofd,"Hashed Tries ===\n");
    fdprintf(ofd,"nblock %d, nline %d, linesize %d (%ldK+%ld), datasize %d (%ldK+%ld)\n",
        nblock, nline, linesize, dK(linesize), mK(linesize),
        datasize, dK(datasize), mK(datasize));
    fdprintf(ofd,"\n");

    for(l=0;l<nblock;l++) {
#if 0
        lr = (line_rec*)&_htrie_slot[nline*linesize*l];
        bcr = (block_cntl_rec*)&lr[0];
#endif
        lr = (line_rec*) cache_block_pos(l);
        bcr = (block_cntl_rec*)&lr[0];


        /* check lockable or not */
        u = 0;
        chk = MUTEX_TRYLOCK(&bcr->mutex);
        if(chk) {
            if(chk==EBUSY) {
                u = 1;
            }
        }
        else {
            MUTEX_UNLOCK(&bcr->mutex);
        }


#if 1
        fdprintf(ofd, "block #%d %p: %s : control: type %d\n",
            l, bcr, u==1 ? "USE" : "-", bcr->type);
#endif
        chk = cache_showblockaddr_fd(l, 0, ofd);


        lp = (char*)bcr + linesize;
        for(w=1;w<nline;w++) {
            lcr = (line_cntl_rec*) lp;

#if 0
fdprintf(ofd, " %3d: # %d : %ld %ld\n",
    w, lcr->oid, lcr->size, lcr->length);
#endif

            obj_total_line++;
            if(lcr->size>0) {
                obj_total_areasize += (off_t) lcr->size;
            }
            switch (lcr->type) {
            case TYPE_NORMAL_LINE:
                ncount++;
                if(lcr->length>0) {
                    obj_total_usesize += (off_t) lcr->length;
                }
                break;
            case TYPE_SLAVE_LINE:
                scount++;
                if(lcr->length>0) {
                    obj_total_usesize += (off_t) lcr->length;
                }
                break;
            case TYPE_EMPTY_LINE:
                ecount++;
                break;
            default:
                zcount++;
                break;
            }

            lp += linesize;
        }
#if 0
        fdprintf(ofd, "n %ld, s %ld, e %ld, z %ld//area %lld used %lld\n",
            (long)ncount, (long)scount, (long)ecount, (long)zcount,
            (long long)obj_total_areasize, (long long)obj_total_usesize);
#endif
        fdprintf(ofd, "\n");
    }

    fdprintf(ofd, "LFU ===\n");
    for(w=0;w<lnslot;w++) {
        lr = (line_rec*)&_lfu_slot[w*slotsize];

        obj_total_slot++;
        obj_total_areasize += (off_t) lr->cntl.size;
        if(lr->cntl.type==TYPE_NORMAL_LINE && lr->cntl.length>=0) {
            ncount++;
            obj_total_usesize  += (off_t) lr->cntl.length;
        }

        fdprintf(ofd,"    #%4d: %p : %c %c '%s'(%d) %d %c %d/%d\n",
            w, lr,
            lr->cntl.type==TYPE_NORMAL_LINE ? 'N' :
                (lr->cntl.type==TYPE_SLAVE_LINE ? 'S' : '?'),
            lr->cntl.hot>0 ? 'H' : 'c',
            lr->cntl.name, lr->cntl.lhash,
            lr->cntl.count,
            lr->cntl.cracked ? 'C' : '-',
            lr->cntl.length, lr->cntl.size);
    }

#if 0
    fdprintf(ofd,
        "---\nobject count %ld/%ld, size %ld/%ld (utilization %6.2f%%)\n",
        (long)obj_total_count, (long)obj_total_slot,
        (long)obj_total_usesize, (long)obj_total_areasize,
        ((double) obj_total_usesize * 100.0)/ obj_total_areasize);
#endif

    fdprintf(ofd,
        "---\n\
object cache area\n\
  total:\n\
    count  %ld (line %ld, slot %ld)\n\
    size   %lld (%ldM+%ld)\n\
  used:\n\
    count  %ld (normal %ld, slave %ld)\n\
    size   %lld (%ldM+%ld)\n\
  not used:\n\
    count  %ld (empty %ld, else %ld)\n\
\n\
utilization\n\
  count    %6.2f%%\n\
  size     %6.2f%%\n\
\n", (long)(obj_total_line+obj_total_slot),
        (long)obj_total_line, (long)obj_total_slot,
    (long long)obj_total_areasize,
        (long)dM(obj_total_areasize), (long)mM(obj_total_areasize),
    (long)(ncount + scount), (long)ncount, (long)scount,
    (long long)obj_total_usesize,
        (long)dM(obj_total_usesize), (long)mM(obj_total_usesize),
    (long)ecount, (long)ecount, (long)zcount,
    ((double) ncount* 100.0)/(ncount+scount+ecount+zcount),
    ((double) obj_total_usesize * 100.0)/ obj_total_areasize );

    return 0;
}


void*
cache_dump_all()
{
    int l;
    for(l=0;l<nblock;l++) {
        cache_showblockaddr(l, 0);
    }
}


/*
 * you have to lock before use this routine
 */

void*
cache_find_line(int fh, int sh, char *name)
{
    void* ret;
    char *bpos;
    block_cntl_rec *bcr;
    line_cntl_rec *lcr;
    char *clpos;
    int w;

#if 0
    Trace("cache_find_line: fh %d, sh %d, name '%s'\n",
        fh, sh, name);
#endif

#if 0
    cache_showblock(fh, 0);
#endif

    ret = NULL;

    bpos = (char*) cache_block_pos(fh);
    bcr = (block_cntl_rec*)bpos;

    clpos = bpos + linesize;
    for(w=1;w<nline;w++) {
        lcr = (line_cntl_rec*) clpos;
        if(lcr->type==TYPE_NORMAL_LINE
                && lcr->lhash==sh
                && strcmp((const char*)lcr->name, (const char*)name)==0) {
            ret = (void*)clpos;
            break;
        }
        clpos += linesize;
    }

    return ret;
}



/*
 *
 * return:
 *      found       >0
 *      not found   =0
 *      error       <0
 */
int
cache_have(int fh, int sh, char *name, int flag_check_rcode, time_t reftime)
{
    int ret;
    int chk;
    char *bpos;
    block_cntl_rec *bcr;
    int w;
    line_cntl_rec *lcr;
    char *clpos;
    char *lpos, *fpos;
    int rcode;

#if 0
    Trace("cache_have: fh %d, sh %d, name '%s'\n", fh, sh, name);
#endif


    ret = 0;

    bpos = (char*) cache_block_pos(fh);
    bcr = (block_cntl_rec*)bpos;

#if 0
    Trace(" bpos %p, bcr %p\n", bpos, bcr);
#endif


#if 0
    chk = MUTEX_LOCK(&bcr->mutex);
#endif
    chk = MUTEX_TRYLOCK(&bcr->mutex);
    if(chk) {
        if(chk==EBUSY) {
#if 0
            Error("cache_have: block #%d is busy [%d:%d]", fh, fh, sh);
#endif
            ret = -1;
        }
        else {
            Error("cache_have: fail locking block #%d (%d)", fh, chk);
            ret = -2;
        }
        goto out;
    }

    lpos = NULL;
    fpos = NULL;

    lpos = (char*) cache_find_line(fh, sh, name);
    lcr = (line_cntl_rec*) lpos;

    /* found specified object */
    if(lpos!=NULL) {
        if(lcr->cracked) {
#if 0
            Trace("  object cache was CRACKED\n");
#endif
            ret = 0;
        }
        else {
#if 0
            Trace("  FOUND in object cache\n");
#endif

            if(flag_check_rcode) {
                rcode = HTTP_parse_response_code((char*)&((line_rec*)lcr)->body,
                            lcr->length);

                /* 2xx-3xx */
                if(rcode>=200 && rcode<=399) {
                    if(reftime>0)
                       lcr->hot += reftime;
                    ret = 1;
                }
                /* 4xx-5xx */
                else {
#if 0
    Trace("  FOUND in object cache, but its response code is %d\n", rcode);
#endif
                    ret = 0;
                }
            }
            else {
                ret = 1;
            }

#if 0
            ret = 1;
#endif
        }
    }
#if 0
    else {
        Trace("  not found in object cache\n");
    }
#endif

    MUTEX_UNLOCK(&bcr->mutex);

out:

    return ret;
}



int
cache_list_overview_HTML(int ofd)
{
    int    cw;  /* current */
    int    sw;  /* summing */

    off_t act_htrie_area;
    off_t act_lfu_area;

    size_t psize;
    size_t lcsize;

    sw = 0;

    cw = fdprintf(ofd, "<h2>Cache Overview</h2>\n\n");
    if(cw>0) sw += cw;


    /* Total */

    cw = fdprintf(ofd, "<h3>Summary</h3>\n\n");
    if(cw>0) sw += cw;

    lcsize = sizeof(line_cntl_rec);
    act_htrie_area = nblock * (nline - 1) * datasize;
    act_lfu_area = lnslot * ldatasize;


    cw = fdprintf(ofd,
      "<table border=\"0\" cellspacing=\"0\" cellpadding=\"4\">\n\n");
    if(cw>0) sw += cw;
    cw = fdprintf(ofd, "  <tr bgcolor=\"#ffffcc\"><th>-<th>bytes<th>Mbytes\n");
    if(cw>0) sw += cw;

#ifdef HAVE_QUAD_T

    cw =
      fdprintf(ofd, "  <tr><th bgcolor=\"#ffcccc\">Total <td align=\"right\">%qd <td>%qdM+%qd\n\n",
        (quad_t)cachetotal,
        (quad_t)dM(cachetotal), (quad_t)mM(cachetotal));
    if(cw>0) sw += cw;
    cw =
      fdprintf(ofd, "  <tr><th bgcolor=\"#ffcccc\">Hashed Tries<td align=\"right\">%qd <td>%qdM+%qd\n",
        (quad_t)htrie_area_size,
        (quad_t)dM(htrie_area_size), (quad_t)mM(htrie_area_size));
    if(cw>0) sw += cw;
    cw =
      fdprintf(ofd, "  <tr><th bgcolor=\"#ffcccc\">LFU   <td align=\"right\">%qd <td>%qdM+%qd\n",
        (quad_t)lfu_area_size,
        (quad_t)dM(lfu_area_size), (quad_t)mM(lfu_area_size));
    if(cw>0) sw += cw;

#else

    cw =
      fdprintf(ofd, "  <tr><th bgcolor=\"#ffcccc\">Total <td>%lld <td>%lldM+%lld\n\n",
        (long long)cachetotal,
        (long long)dM(cachetotal), (long long)mM(cachetotal));
    if(cw>0) sw += cw;
    cw =
      fdprintf(ofd, "  <tr><th bgcolor=\"#ffcccc\">Hased Tries<td align=\"right\">%lld <td>%lldM+%lld\n",
        (long long)htrie_area_size,
        (long long)dM(htrie_area_size), (long long)mM(htrie_area_size));
    if(cw>0) sw += cw;
    cw =
      fdprintf(ofd, "  <tr><th bgcolor=\"#ffcccc\">LFU   <td align=\"right\">%lld <td>%lldM+%lld\n",
        (long long)lfu_area_size,
        (long long)dM(lfu_area_size), (long long)mM(lfu_area_size));
    if(cw>0) sw += cw;

#endif

    cw = fdprintf(ofd, "</table>\n\n");
    if(cw>0) sw += cw;


    /***
     *** Hashed Trie
     ***/

    if(nblock==0) {
        Error("nblock is 0\n");
    }


    cw = fdprintf(ofd, "<h3>Hashed Tries</h3><p>\n");
    if(cw>0) sw += cw;

    cw = fdprintf(ofd, "    number of blocks = %ld<br>\n", (long)nblock);
    if(cw>0) sw += cw;
    cw = fdprintf(ofd,
        "    the number of lines = %ld (one line is used as block control area)<br>\n",
        (long)nline);
    if(cw>0) sw += cw;

    cw = fdprintf(ofd, "    size of line = %d (%dK+%d)<br>\n\n",
        linesize, dK(linesize), mK(linesize));
    if(cw>0) sw += cw;
    cw = fdprintf(ofd, "    size of control block in line = %d (%dK+%d)<br>\n",
        lcsize, dK(lcsize), mK(lcsize));
    if(cw>0) sw += cw;
    cw = fdprintf(ofd, "    size of data in line = %d (%dK+%d)<br>\n",
        datasize, dK(datasize), mK(datasize));
    if(cw>0) sw += cw;

#ifdef HAVE_QUAD_T
    cw = fdprintf(ofd, "    <b>actual size = %qd (%ldM+%ld)</b>\n\
= %d x %d x %d<br>\n",
        (quad_t)act_htrie_area, (long)dM(act_htrie_area), (long)mM(act_htrie_area),
        nblock, (nline - 1), datasize
    );
#else
    cw = fdprintf(ofd, "    <b>actual size = %lld (%ldM+%ld)</b>\n\
= %d x %d x %d<br>\n",
        (long long)act_htrie_area, (long)dM(act_htrie_area), (long)mM(act_htrie_area),
        nblock, (nline - 1), datasize
    );
#endif
    if(cw>0) sw += cw;


    cw = fdprintf(ofd, "\n");
    if(cw>0) sw += cw;



    /***
     *** LFU
     ***/

    cw = fdprintf(ofd, "<h3>LFU</h3><p>\n");
    if(cw>0) sw += cw;


    cw = fdprintf(ofd, "number of slots = %d<br>\n", lnslot);
    if(cw>0) sw += cw;

#ifdef HAVE_QUAD_T
    cw = fdprintf(ofd, "size of slot = %qd (%ldM+%ld)<br>\n",
        (quad_t)slotsize,
        (long)dM(slotsize), (long)mM(slotsize));
    if(cw>0) sw += cw;
    cw = fdprintf(ofd, "size of data in slot = %qd (%ldM+%ld)<br>\n",
        (quad_t)ldatasize,
        (long)dM(ldatasize), (long)mM(ldatasize));
    if(cw>0) sw += cw;
#else
    cw = fdprintf(ofd, "size of slot = %lld (%ldM+%ld)<br>\n",
        (long long)slotsize,
        (long)dM(slotsize), (long)mM(slotsize));
    if(cw>0) sw += cw;
    cw = fdprintf(ofd, "size of data in slot = %lld (%ldM+%ld)<br>\n",
        (long long)ldatasize,
        (long)dM(ldatasize), (long)mM(ldatasize));
    if(cw>0) sw += cw;
#endif

#ifdef HAVE_QUAD_T
    cw = fdprintf(ofd, "    <b>actual size = %qd (%ldM+%ld)</b>\n\
= %d x %qd<br>\n",
        (quad_t)act_lfu_area, (long)dM(act_lfu_area), (long)mM(act_lfu_area),
        lnslot, (quad_t)ldatasize);
#else
    cw = fdprintf(ofd, "    <b>actual size = %lld (%ldM+%ld)</b>\n\
= %d x %lld<br>\n",
        (long long)act_lfu_area, (long)dM(act_lfu_area), (long)mM(act_lfu_area),
        lnslot, (long long)ldatasize);
#endif
    if(cw>0) sw += cw;



    /***
     *** Others
     ***/

    psize = getpagesize();
    cw = fdprintf(ofd, "<h3>System Propaties</h3>\n\
<p>page size of this system  %d bytes(%dK+%d)<br>\n\n",
        psize, dK(psize), mK(psize));
    if(cw>0) sw += cw;

    cw = fdprintf(ofd, "</p>\n");
    if(cw>0) sw += cw;

    Trace("write %d bytes to fd #%d\n", sw, ofd);

    return sw;
}



