/*
 * hostname caching procedures for WWW proxies
 *      by k-chinen@is.aist-nara.ac.jp, 1999, 2000
 *
 * $Id$
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <limits.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <assert.h>
#include <errno.h>


#if 0
#include "wcol.h"
#endif
#include "range.h"
#include "socket.h"
#include "misc.h"

/*
#include "cache.h"
*/
#include "sim_stat.h"

#ifndef HOSTQUERY_PORT
#define HOSTQUERY_PORT      (1245)
#endif

#define STAT_LIFETIME   (1024*1024*1024)
#define STAT_REPORT     (20)
#define STAT_LEN        (1024)


#define NSLOT       (79)


#define _TRACE_HCACHE
#define _TRACE_DETAIL_HCACHE



/* static */ sim_rec hres_stat;
long hres_stat_count[STAT_LEN];

struct sockaddr_in resolver_addr;


pthread_mutex_t hcache_lock = PTHREAD_MUTEX_INITIALIZER;
char hname_slot[NSLOT][L_HOST];
struct sockaddr_in haddr_slot[NSLOT];
int slot_use[NSLOT];


static
unsigned
int
hhash(unsigned char *b)
{
    register unsigned int sum;
    register unsigned char *p;

    sum = 0;
    p = b;
    while(*b) {
        sum = (sum<<1) + *b;
        b++;
    }

    return sum % NSLOT;
}


int
hcache_setup()
{
    int chk;
    int i;

    chk = addr_byname("localhost", HOSTQUERY_PORT, &resolver_addr);
    if(chk) {
        Log("; resolver not found");
        return -1;
    }

    sim_stat_alloc_new(&hres_stat, "hostname resolve time",
        STAT_LEN, hres_stat_count);

    MUTEX_LOCK(&hcache_lock);
    for(i=0;i<NSLOT;i++) {
        hname_slot[i][0] = '\0';
        slot_use[i] = 0;
    }
    MUTEX_UNLOCK(&hcache_lock);

    return 0;
}

void
hcache_list()
{
    int i;

    MUTEX_LOCK(&hcache_lock);
    fflush(stdout);
    fprintf(stderr, "=== hostname cache ===\n");
    for(i=0;i<NSLOT;i++) {
        fprintf(stderr, "%4d: %12d '%s'\n", i, slot_use[i], hname_slot[i]);
    }
    fflush(stderr);
    MUTEX_UNLOCK(&hcache_lock);
}


extern char *_htrie_slot, *htrie_shadow_top;

int
hcache_got_addr_direct(char *name, struct sockaddr_in* addr)
{
    int nr, nw;
    int chk;
    int ret;
    char protocol[L_URL], portstr[L_URL];
    int qfd;
    char addrbuf[32];

#ifdef TRACE_HCACHE
    Trace("hcache_got_addr_direct: name '%s'\n", name);
#endif

    if(name==NULL || *name=='\0') {
        return -1;
    }
#ifdef TRACE_HCACHE
    Trace("hcache_got_addr_direct: name '%s'\n", name);
#endif

 assert(_htrie_slot==htrie_shadow_top);

    ret = 0;

    if(*name>='0' && *name<='9') {
        register char *p;

        /* check, name is filled digits and/or dots */
        p = name;
        while(*p && ((*p>='0' && *p<='9') || *p=='.')) {
            p++;
        }

        if(!*p) {
            memset(addr, 0, sizeof(addr));
            addr->sin_family = AF_INET;
            addr->sin_port   = 0;
            addr->sin_addr.s_addr = inet_addr(name);

            goto out;
        }
    }

 assert(_htrie_slot==htrie_shadow_top);

    /***
     *** name resolving
     ***/
    qfd = TCP_connect_addr(&resolver_addr, HOSTQUERY_PORT);
 assert(_htrie_slot==htrie_shadow_top);
    if(qfd<0) {
        Error("fail TCP_connect_addr() ret %d (%d)", qfd, errno);
        ret = -2;
        goto out;
    }
#ifdef TRACE_HCACHE
    Trace("qfd %d, query host-'%s' %d\n",
        qfd, name, strlen(name)+1);
#endif

try_write:
 assert(_htrie_slot==htrie_shadow_top);
    nw = write(qfd, name, strlen(name)+1);
#ifdef TRACE_HCACHE
    Trace("write %d bytes (%d)\n", nw, errno);
#endif
    if(nw<0) {
        if(errno==EINTR) {
            goto try_write;
        }
        Error("write() ret %d (%d)", nw, errno);
        ret = -3;
        goto try_close;
    }

try_read:
 assert(_htrie_slot==htrie_shadow_top);
    nr = read(qfd, addrbuf, 32);
#ifdef TRACE_HCACHE
    Trace("read %d bytes (%d)\n", nr, errno);
#endif
    if(nr<0) {
        if(errno==EINTR) {
            goto try_read;
        }
        Error("read() ret %d (%d)", nr, errno);
        ret = -4;
        goto try_close;
    }
    else
    if(nr>0) {
        addr->sin_family = AF_INET;
        memcpy(&(addr->sin_addr.s_addr), addrbuf, 4);
    }
    else {
        ret = -5;
    }

try_close:
 assert(_htrie_slot==htrie_shadow_top);
    shutdown(qfd, 2);
    close(qfd);

out:
 assert(_htrie_slot==htrie_shadow_top);
    return ret;
}


int
hcache_send_query_addr_direct(char *name, struct sockaddr_in* addr, int *qfd)
{
    int nr, nw;
    int chk;
    int ret;
    char protocol[L_URL], portstr[L_URL];
    char addrbuf[32];

#ifdef TRACE_HCACHE
    Trace("hcache_send_query_addr_direct: name '%s'\n", name);
#endif
 assert(_htrie_slot==htrie_shadow_top);

    if(name==NULL || *name=='\0') {
        return -1;
    }

    ret = 0;

    if(*name>='0' && *name<='9') {
        register char *p;

        /* check, name is filled digits and/or dots */
        p = name;
        while(*p && ((*p>='0' && *p<='9') || *p=='.')) {
            p++;
        }

        if(!*p) {
            memset(addr, 0, sizeof(addr));
            addr->sin_family = AF_INET;
            addr->sin_port   = 0;
            addr->sin_addr.s_addr = inet_addr(name);

            goto out;
        }
    }

    /***
     *** name resolving
     ***/
    *qfd = TCP_connect_addr(&resolver_addr, HOSTQUERY_PORT);
    if(*qfd<0) {
        Error("fail TCP_connect_addr() ret %d (%d)", *qfd, errno);
        ret = -2;
        goto out;
    }
#ifdef TRACE_HCACHE
    Trace("  query: fd %d, host '%s', length %d\n",
        *qfd, name, strlen(name)+1);
#endif

try_write:
    nw = write(*qfd, name, strlen(name)+1);
#ifdef TRACE_HCACHE
    Trace("  write %d bytes (%d)\n", nw, errno);
#endif
    if(nw<0) {
        if(errno==EINTR) {
            goto try_write;
        }
        Error("write() ret %d (%d)", nw, errno);
        ret = -3;

        shutdown(*qfd, 2);
        close(*qfd);
        *qfd = -1;

        goto out;
    }

out:
 assert(_htrie_slot==htrie_shadow_top);
    return ret;
}

/*
 * success  0
 * fail     !0
 *
 */
int
hcache_recv_response_addr_direct(char *name, struct sockaddr_in* addr, int qfd)
{
    int nr, nw;
    int chk;
    int ret;
    char protocol[L_URL], portstr[L_URL];
    char addrbuf[32];

    ret = 0;

#ifdef TRACE_HCACHE
    Trace("hcache_recv_response_addr_direct: name '%s'\n", name);
#endif
 assert(_htrie_slot==htrie_shadow_top);

    if(name==NULL || *name=='\0') {
        return -1;
    }
    if(qfd<0) {
        return -1;
    }

try_read:
    nr = read(qfd, addrbuf, 32);
#ifdef TRACE_HCACHE
    Trace("  read %d bytes (%d)\n", nr, errno);
#endif
    if(nr<0) {
        if(errno==EINTR) {
            goto try_read;
        }
        Error("read() ret %d (%d)", nr, errno);
        ret = -4;
        goto try_close;
    }
    else
    if(nr>0) {
        addr->sin_family = AF_INET;
        memcpy(&(addr->sin_addr.s_addr), addrbuf, 4);
    }
    else {
        Error("read() ret 0");

        ret = -5;
    }

try_close:
    shutdown(qfd, 2);
    close(qfd);

out:
 assert(_htrie_slot==htrie_shadow_top);
    return ret;
}


int
hcache_seek_addr_incache(char *name, struct sockaddr_in* addr)
{
    int chk;
    int ret;
    char protocol[L_URL], portstr[L_URL];
    int qfd;
    char addrbuf[32];
    int i, h, s, pos, free_pos, min, min_pos;
    time_t now;

#ifdef TRACE_HCACHE
    Trace("hcache_got_addr_wcache: name '%s'\n", name);
#endif

 assert(_htrie_slot==htrie_shadow_top);
    if(name==NULL || *name=='\0') {
        return -1;
    }

    ret = -1;


    /***
     *** check numeric name
     ***/
    if(*name>='0' && *name<='9') {
        register char *p;

        /* check, name is filled digits and/or dots */
        p = name;
        while(*p && ((*p>='0' && *p<='9') || *p=='.')) {
            p++;
        }

        if(!*p) {
            memset(addr, 0, sizeof(addr));
            addr->sin_family = AF_INET;
            addr->sin_port   = 0;
            addr->sin_addr.s_addr = inet_addr(name);

            ret = 0;
            goto out;
        }
    }


    /***
     *** check cache
     ***/

 assert(_htrie_slot==htrie_shadow_top);
    h = (int)hhash((unsigned char*)name);
    now = time(NULL);

#ifdef TRACE_DETAIL_HCACHE
    fprintf(stderr, "    *** hash value %d\n", h);

    MUTEX_LOCK(&hcache_lock);
    for(i=0;i<NSLOT;i++) {
 assert(_htrie_slot==htrie_shadow_top);
        fprintf(stderr, "%4d: %12d '%s'\n", i, slot_use[i], hname_slot[i]);
    }
    MUTEX_UNLOCK(&hcache_lock);
#endif

    MUTEX_LOCK(&hcache_lock);


    /*
     * Search same name, free position and minimum value postion in
     * once sweep.  However, if I check all entry, it will be liner search.
     * Then, this routine check in short range (quoater) per one sweep.
     */

    pos = -1;
    free_pos = -1;
    min = INT_MAX;
    min_pos = -1;
    i = 0;
    s = h;
    while(i<NSLOT/4) {
 assert(_htrie_slot==htrie_shadow_top);
        if(hname_slot[s][0] == *name && strcmp(hname_slot[s], name)==0) {
            pos = s;
            break;
        }
        if(hname_slot[s][0] == '\0') {
            free_pos = s;
            break;
        }
 assert(_htrie_slot==htrie_shadow_top);

        if(slot_use[s]<min) {
            min = slot_use[s];
            min_pos = s;
        }

        i++;
        s = (s+1)%NSLOT;
    }


 assert(_htrie_slot==htrie_shadow_top);
    /*
     * found
     */
    if(pos>=0) {
#ifdef TRACE_HCACHE
        Trace("  found in hostname cache.\n");
#endif

 assert(_htrie_slot==htrie_shadow_top);
        memcpy(addr, &haddr_slot[pos], sizeof(struct sockaddr_in));
 assert(_htrie_slot==htrie_shadow_top);

/*
        slot_use[pos]++;
*/
        slot_use[pos] = (int)now;
 assert(_htrie_slot==htrie_shadow_top);

        ret = 0;
        goto out;
    }

out:
 assert(_htrie_slot==htrie_shadow_top);
    MUTEX_UNLOCK(&hcache_lock);
    return ret;
}



/* with cache */
int
hcache_set_addr_incache(char *name, struct sockaddr_in* addr)
{
    int chk;
    int ret;
    char protocol[L_URL], portstr[L_URL];
    int qfd;
    char addrbuf[32];
    int i, h, s, pos, free_pos, min, min_pos;
    time_t now;

#ifdef TRACE_HCACHE
    Trace("hcache_got_addr_incache: name '%s'\n", name);
#endif

 assert(_htrie_slot==htrie_shadow_top);
    if(name==NULL || *name=='\0') {
        return -1;
    }

    ret = -1;


    /***
     *** check cache
     ***/

 assert(_htrie_slot==htrie_shadow_top);
    h = (int)hhash((unsigned char*)name);
    now = time(NULL);

#ifdef TRACE_DETAIL_HCACHE
    fprintf(stderr, "    *** hash value %d\n", h);

    MUTEX_LOCK(&hcache_lock);
    for(i=0;i<NSLOT;i++) {
 assert(_htrie_slot==htrie_shadow_top);
        fprintf(stderr, "%4d: %12d '%s'\n", i, slot_use[i], hname_slot[i]);
    }
    MUTEX_UNLOCK(&hcache_lock);
#endif

    MUTEX_LOCK(&hcache_lock);


    /*
     * Search same name, free position and minimum value postion in
     * once sweep.  However, if I check all entry, it will be liner search.
     * Then, this routine check in short range (quoater) per one sweep.
     */

    pos = -1;
    free_pos = -1;
    min = INT_MAX;
    min_pos = -1;
    i = 0;
    s = h;
    while(i<NSLOT/4) {
 assert(_htrie_slot==htrie_shadow_top);
        if(hname_slot[s][0] == *name && strcmp(hname_slot[s], name)==0) {
            pos = s;
            break;
        }
        if(hname_slot[s][0] == '\0') {
            free_pos = s;
            break;
        }
 assert(_htrie_slot==htrie_shadow_top);

        if(slot_use[s]<min) {
            min = slot_use[s];
            min_pos = s;
        }

        i++;
        s = (s+1)%NSLOT;
    }


 assert(_htrie_slot==htrie_shadow_top);
    /*
     * found
     */
    if(pos>=0) {
#ifdef TRACE_HCACHE
        Trace("  found in hostname cache.\n");
#endif

 assert(_htrie_slot==htrie_shadow_top);
#if 0
        memcpy(addr, &haddr_slot[pos], sizeof(struct sockaddr_in));
#endif
        memcpy(&haddr_slot[pos], addr, sizeof(struct sockaddr_in));
 assert(_htrie_slot==htrie_shadow_top);

/*
        slot_use[pos]++;
*/
        slot_use[pos] = (int)now;
 assert(_htrie_slot==htrie_shadow_top);

        ret = 0;
        goto out;
    }


#ifdef TRACE_HCACHE
    /*
    Trace("  not found in hostname cache, retrieve from other system.\n");
    */
    Trace("  still it is not exist\n");
#endif

    /*
     * not found
     */
    if(free_pos==-1) {
        if(min_pos==-1) {
 assert(_htrie_slot==htrie_shadow_top);
            free_pos = h;
        }
        else {
 assert(_htrie_slot==htrie_shadow_top);
            free_pos = min_pos;
        }
    }


    /* cachable */
    if(strlen(name)<L_HOST-1) {
        strncpy(hname_slot[free_pos], name, L_HOST);
        memcpy(&haddr_slot[free_pos], addr, sizeof(struct sockaddr_in));
        slot_use[free_pos] = (int)now;
    }

    ret = 0;


out:
 assert(_htrie_slot==htrie_shadow_top);
    MUTEX_UNLOCK(&hcache_lock);
    return ret;
}


/* with cache */
int
hcache_got_addr_wcache(char *name, struct sockaddr_in* addr)
{
    int chk;
    int ret;
    char protocol[L_URL], portstr[L_URL];
    int qfd;
    char addrbuf[32];
    int i, h, s, pos, free_pos, min, min_pos;
    time_t now;

#ifdef TRACE_HCACHE
    Trace("hcache_got_addr_wcache: name '%s'\n", name);
#endif

 assert(_htrie_slot==htrie_shadow_top);
    if(name==NULL || *name=='\0') {
        return -1;
    }

    ret = -1;


    /***
     *** check cache
     ***/

 assert(_htrie_slot==htrie_shadow_top);
    h = (int)hhash((unsigned char*)name);
    now = time(NULL);

#ifdef TRACE_DETAIL_HCACHE
    fprintf(stderr, "    *** hash value %d\n", h);

    MUTEX_LOCK(&hcache_lock);
    for(i=0;i<NSLOT;i++) {
 assert(_htrie_slot==htrie_shadow_top);
        fprintf(stderr, "%4d: %12d '%s'\n", i, slot_use[i], hname_slot[i]);
    }
    MUTEX_UNLOCK(&hcache_lock);
#endif

    MUTEX_LOCK(&hcache_lock);


    /*
     * Search same name, free position and minimum value postion in
     * once sweep.  However, if I check all entry, it will be liner search.
     * Then, this routine check in short range (quoater) per one sweep.
     */

    pos = -1;
    free_pos = -1;
    min = INT_MAX;
    min_pos = -1;
    i = 0;
    s = h;
    while(i<NSLOT/4) {
 assert(_htrie_slot==htrie_shadow_top);
        if(hname_slot[s][0] == *name && strcmp(hname_slot[s], name)==0) {
            pos = s;
            break;
        }
        if(hname_slot[s][0] == '\0') {
            free_pos = s;
            break;
        }
 assert(_htrie_slot==htrie_shadow_top);

        if(slot_use[s]<min) {
            min = slot_use[s];
            min_pos = s;
        }

        i++;
        s = (s+1)%NSLOT;
    }


 assert(_htrie_slot==htrie_shadow_top);
    /*
     * found
     */
    if(pos>=0) {
#ifdef TRACE_HCACHE
        Trace("  found in hostname cache.\n");
#endif

 assert(_htrie_slot==htrie_shadow_top);
        memcpy(addr, &haddr_slot[pos], sizeof(struct sockaddr_in));
 assert(_htrie_slot==htrie_shadow_top);

/*
        slot_use[pos]++;
*/
        slot_use[pos] = (int)now;
 assert(_htrie_slot==htrie_shadow_top);

        ret = 0;
        goto out;
    }


#ifdef TRACE_HCACHE
    Trace("  not found in hostname cache, retrieve from other system.\n");
#endif

    /*
     * not found
     */
    if(free_pos==-1) {
        if(min_pos==-1) {
 assert(_htrie_slot==htrie_shadow_top);
            free_pos = h;
        }
        else {
 assert(_htrie_slot==htrie_shadow_top);
            free_pos = min_pos;
        }
    }

 assert(_htrie_slot==htrie_shadow_top);
    chk = hcache_got_addr_direct(name, addr);
 assert(_htrie_slot==htrie_shadow_top);

    if(chk==0) {
        /* success */

        /* cachable */
        if(strlen(name)<L_HOST-1) {
 assert(_htrie_slot==htrie_shadow_top);
            strncpy(hname_slot[free_pos], name, L_HOST);
 assert(_htrie_slot==htrie_shadow_top);
            memcpy(&haddr_slot[free_pos], addr, sizeof(struct sockaddr_in));
 assert(_htrie_slot==htrie_shadow_top);
/*
            slot_use[free_pos] = 1;
*/
            slot_use[free_pos] = (int)now;
        }
 assert(_htrie_slot==htrie_shadow_top);

        ret = 0;
    }
    else {
 assert(_htrie_slot==htrie_shadow_top);
        Error("cannot get hostname [%d]", chk);
        ret = chk;
 assert(_htrie_slot==htrie_shadow_top);
    }


out:
 assert(_htrie_slot==htrie_shadow_top);
    MUTEX_UNLOCK(&hcache_lock);
    return ret;
}



int
hcache_got_addr(char *name, struct sockaddr_in* addr)
{
    int ret;
#if 1
    struct timeval before, after;
#else
    struct timespec before, after;
#endif
    long cv;

 assert(_htrie_slot==htrie_shadow_top);

    if(hres_stat.count>STAT_LIFETIME) {
        sim_stat_reset(&hres_stat);
    }

 assert(_htrie_slot==htrie_shadow_top);

#if 1
    gettimeofday(&before, NULL);
    ret = hcache_got_addr_wcache(name, addr);
    gettimeofday(&after, NULL);
#else
    clock_gettime(CLOCK_REALTIME, &before);
    ret = hcache_got_addr_wcache(name, addr);
    clock_gettime(CLOCK_REALTIME, &after);
#endif

 assert(_htrie_slot==htrie_shadow_top);

#if 1
#if 0
    cv = ((after.tv_sec%3600)*1000+after.tv_usec/1000)
            -((before.tv_sec%3600)*1000+before.tv_usec/1000) ;
#endif
    cv = ((after.tv_sec%3600)*1000000+after.tv_usec)
            -((before.tv_sec%3600)*1000000+before.tv_usec) ;
#else
    cv = ((after.tv_sec%3600)*1000+after.tv_usec/1000)
            -((before.tv_sec%3600)*1000+before.tv_usec/1000) ;
#endif

    sim_stat_add_value(&hres_stat, cv);

#if 0
    if(hres_stat.count%STAT_REPORT==0) {
        sim_stat_show_lock(&hres_stat, stderr);
    }
#endif

 assert(_htrie_slot==htrie_shadow_top);

    return ret;
}
