/*
 * hostname resolver
 *          by k-chinen@is.aist-nara.ac.jp, 1999
 *
 * $Id$
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <signal.h>
#include <setjmp.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <errno.h>


#include "range.h"
#include "msg.h"
#include "socket.h"
#include "sigs.h"

#include "sim_stat.h"
#include "misc.h"


#define DEFAULT_HRES_SERV_PORT  (1245)

#define HRES_MAX_FORK   (32)

#define STAT_INTERVAL   (1000000)
#define STAT_REPORT     (1000)
#define STAT_LEN        (1024)



sim_rec hres_stat;
long hres_stat_count[KB(1)];




static
int
slave_loop(int listen_fd)
{
    pid_t cpid;

    int chk;
    char buf[L_BUF];
    struct hostent *hp;
    char *addrstr;
    int nread, nwrite;
    int cfd;
    struct sockaddr_in from;
    struct in_addr target_addr;
    int from_len;
    struct timeval before, after;
    long cv;


    int i;
    int ntry;

    Trace("actual resolver pid# %d\n", getpid());

    while(1) {
        if(hres_stat.count>STAT_INTERVAL) {
            sim_stat_reset(&hres_stat);
        }

        from_len = sizeof(from);
try_accept:
        cfd = accept(listen_fd, (struct sockaddr*)&from, &from_len);
        if(cfd<0) {
            if(errno==EINTR)
                goto try_accept;
            Error("errno %d", errno);
            continue;
        }

/*
        before = time(NULL);
*/
        gettimeofday(&before, NULL);
        nread = nwrite = -1;

try_read:
        nread = read(cfd, buf, L_BUF-1);
#if 0
        fprintf(stderr, "read [%s] %d bytes\n", buf, nread);
#endif
        if(nread<0) {
            if(errno==EINTR) {
                goto try_read;
            }
#if 0
            fprintf(stderr, "cannot read request\n");
#endif
            goto done;
        }

        buf[nread] = '\0';

        /* chop CR or LF */
        {
            register char *p;
            p = buf;
            while(*p) {
                if(*p=='\r' || *p=='\n' || *p==' ' || *p=='\t') {
                    *p='\0';
                    break;
                }
                p++;
            }
        }

#if 0
        fprintf(stderr, "target hostname '%s'... ", buf);
#endif
        hp = gethostbyname(buf);
        if(hp==NULL) {
            addrstr = NULL;
#if 0
            fprintf(stderr, "cannot resolve");
#endif
        }
        else {
            memcpy(&target_addr, hp->h_addr_list[0], hp->h_length);
            nwrite = write(cfd,  hp->h_addr, hp->h_length);

#if 0
            fprintf(stderr, "solved as '%s'", inet_ntoa(target_addr));
#endif
        }

done:
        shutdown(cfd, 2);
        close(cfd);

        gettimeofday(&after, NULL);

        cv = ((after.tv_sec%3600)*1000+after.tv_usec/1000)
                -((before.tv_sec%3600)*1000+before.tv_usec/1000) ;

#if 0
        fprintf(stderr, " [%d msec]\n", cv);
#endif

        sim_stat_add_value(&hres_stat, cv);


#if 0
        if(hres_stat.count%STAT_REPORT==0) {
            sim_stat_show(&hres_stat, stderr);
        }
        else {
            sim_stat_show_short(&hres_stat, stderr);
        }
#endif


    }

}

int   nchildlen;
pid_t childlenpids[HRES_MAX_FORK];

RETSIGTYPE
childlen_killer()
{
    int i;

    for(i=0;i<nchildlen;i++) {
        if(childlenpids[i]>0) {
            Trace("kill child #%d\n", childlenpids[i]);
            kill(childlenpids[i], SIGHUP);
        }
    }

    exit(0);
}

RETSIGTYPE
childlen_reaper()
{
    pid_t pid;
    int stat;
    int i;

    do {
try_waitpid:
        pid = waitpid((pid_t)-1, &stat, WNOHANG | WUNTRACED);

        if(pid < 0) {
            if(errno==EINTR) {
                goto try_waitpid;
            }
            Trace("error with errno=%d\n", errno);
            break;
        }
        if (pid > 0) {
            Trace("child %d terminated\n", pid);
            for(i=0;i<nchildlen;i++) {
                if(childlenpids[i]==pid) {
                    childlenpids[i] = -1;
                }
            }

            nchildlen--;
        }
    } while(pid>0);

    return;
}

int listen_fd;

static
int
master_loop(int port, int nfork)
{
    pid_t cpid;

    int chk;
    char buf[L_BUF];
    struct hostent *hp;
    char *addrstr;
    int nread, nwrite;
    int cfd;
    struct sockaddr_in from;
    struct in_addr target_addr;
    int from_len;
    struct timeval before, after;
    long cv;


    int pos;
    int ntry;


    Trace("hostname resolver start, port %d, # of fork = %d\n",
        port, nfork);


    /* boot listener */
    listen_fd = TCP_bind_port(port, -1);
    Trace("listen_fd %d\n", listen_fd);
    if(listen_fd<0) {
        Error("port is not free");
        return -1;
    }

    sim_stat_alloc_new(&hres_stat, "retrieval time [msec]",
        KB(1), hres_stat_count);

    if(nfork<0) {
        return -1;
    }
    if(nfork==0) {
        nfork = 1;
    }
    if(nfork>HRES_MAX_FORK) {
        nfork = HRES_MAX_FORK;
    }

    for(pos=0;pos<HRES_MAX_FORK;pos++) {
        childlenpids[pos] = -1;
    }


    /*
     * main loop
     */
    Trace("master process #%d\n", getpid());

    nchildlen = 0;
    while(1) {
        while(nchildlen<nfork) {
            Trace("%d child (required %d)\n", nchildlen, nfork);

            for(pos=0;pos<nfork;pos++) {
                if(childlenpids[pos]>0) {
                    continue;
                }

                Trace("launch %d th slave\n", pos);
                ntry = 4;
                do {
                    cpid = fork();

                    /* error */
                    if(cpid<0) {
                        Error("fail fork() %d times", ntry);
                    }
                    else
                    /* parent */
                    if(cpid>0) {
                        Trace("process #%d, %d, boot child #%d\n",
                            getpid(), ntry, cpid);

                        childlenpids[pos] = cpid;
                        nchildlen++;
                        break;
                    }
                    /* child */
                    else {
                        signal(SIGHUP, SIG_DFL);

                        slave_loop(listen_fd);
                        exit(0);
                    }
                    ntry--;
                } while(ntry>0);
            }

#if 1
            /* print slave's pid list */
            Trace(" # pid\n", pos, childlenpids[pos]);
            for(pos=0;pos<nfork;pos++) {
                Trace("%2d %d\n", pos, childlenpids[pos]);
            }
#endif
        }
        sleep(10);
    }

}

int
main(int argc, char **argv)
{
    int chk;
    int nchild;

    nchild = 1;

    TraceOff();

    if(argc>1) {
        nchild = atoi(argv[1]);
        if(nchild<=0) {
            nchild = 1;
        }
    }

    SetSignalHandler(SIGHUP, childlen_killer);
    SetSignalHandler(SIGCHLD, childlen_reaper);

    chk = master_loop(DEFAULT_HRES_SERV_PORT, nchild);
    if(chk) {
        exit(3);
    }

    exit(0);
}

