/*
 *  KOTETU: Message and Logging procedures (with mutex of pthread)
 *      by k-chinen@is.aist-nara.ac.jp, 1994, 1995, 1999
 *
 *  $Id: msg.c,v 1.11 1996/11/24 14:48:42 k-chinen Exp k-chinen $
 */

/*
 * Simple:
 *      Trace:  show message for debug.
 *      Log:    write message to log-file. (append newline)
 *
 * Complex:
 *      Error:  message. (append newline)
 *      Fatal:  message and log. (append newline)
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <syslog.h>
#include <pthread.h>
#include <sys/types.h>
#include <fcntl.h>
#include <time.h>
/*
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
*/
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/resource.h>

#include <errno.h>

#include "msg.h"


#define XPID    getpid()
#define XTID    pthread_self()

/*
 * globals
 */
pthread_mutex_t trace_lock = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t log_lock = PTHREAD_MUTEX_INITIALIZER;

/* public */
volatile int _trace_flag_ = 0;      /* it is public, but not use direct */
char log_file[LOG_FILENAMESIZE] = "";

/* private */
static char log_name[LOG_FILENAMESIZE];
static char host_name[LOG_FILENAMESIZE];
static char *mon_str[]={"Jan","Feb","Mar","Apr","May","Jun",
                            "Jul","Aug","Sep","Oct","Nov","Dec"};


static int fd_log = -1;


#ifdef __STDC__
int
InitLogger(char *name)
#else
int
InitLogger(name)
char *name;
#endif
{
    register char *p;

    if(name) {
        strcpy(log_name, name);
    }
    else {
        strcpy(log_name, "kotetu");
    }

    if(gethostname(host_name, LOG_FILENAMESIZE)) {
        strcpy(host_name, "-");
    }
    else {
        /* get short name */
        p = host_name;
        while(*p && *p!='.') {
            p++;
        }
        if(*p=='.')
            *p = '\0';
    }

    fd_log = open(log_file, O_WRONLY|O_CREAT|O_APPEND, 0644);
    if(fd_log < 0 ) {
        fputs("cannot open log file\n", stderr);
    }

    return 0;
}



/*
 * Trace - show message if '_trace_flag_' is up.
 */
void
TraceOn(void)
{
    _trace_flag_ = 1;
}
void
TraceOff(void)
{
    _trace_flag_ = 0;
}
void
TraceTurn(void)
{
    _trace_flag_ = 1 - _trace_flag_;
}

void
TracePush(void)
{
    _trace_flag_--;
}
void
TracePop(void)
{
    _trace_flag_++;
}



/*
 * RawTrace - show message and flush it.
 *          main body of Trace() (see msg.h).
 */

#ifdef __STDC__
int
RawTrace(const char *fmt, ...)
#else
int
RawTrace(va_alist)
va_dcl
#endif
{
#ifndef __STDC__
    char *fmt;
#endif
    va_list args;

#ifdef __STDC__
    va_start(args,fmt);
#else
    va_start(args);
    fmt = va_arg(args, char *);
#endif

    MUTEX_LOCK(&trace_lock);

    vfprintf(stderr,fmt,args);
    fflush(stderr);

    MUTEX_UNLOCK(&trace_lock);

    va_end(args);

    return 0;
}


/*
 * Error - show error messages like as Trace but it is not check '_trace_flag_'.
 *         Because error must be shown.
 */

#ifdef __STDC__
int
Error(const char *fmt, ...)
#else
int
Error(va_alist)
va_dcl
#endif
{
#ifndef __STDC__
    char *fmt;
#endif
    va_list args;
    char report[LOG_LINESIZE];

#ifdef __STDC__
    va_start(args,fmt);
#else
    va_start(args);
    fmt = va_arg(args, char *);
#endif

    vsprintf(report, fmt, args);

    MUTEX_LOCK(&trace_lock);

    fprintf(stderr, "### ERROR %d:%08x> %s\n", XPID, XTID, report);
    fflush(stderr);

    MUTEX_UNLOCK(&trace_lock);

    va_end(args);

    return 0;
}


/*
 * RawLog --- logging procedure; write line to log-file or send to syslog
 */
static
int
RawLog(int level, char *rline)
{
#ifdef USE_SYSLOG
    MUTEX_LOCK(&log_lock);
    syslog(level, "%s", rline);
    MUTEX_UNLOCK(&log_lock);
#else

    int log_fd;
    time_t now;
    struct tm *ntm;
    char line[LOG_LINESIZE];

    if(rline[0]=='\0') {
        fprintf(stderr, "RawLog: report is empty\n");
        return -1;
    }

    time(&now);
#ifdef LOG_LOCALTIME
    ntm = localtime(&now);      /* get time with local time */
#else
    ntm = gmtime(&now);         /* get time with UTC (a.k.a. GMT) */
#endif

    MUTEX_LOCK(&log_lock);

    /* open log file. if it is fail, try to open console */
    if((log_fd=open(log_file, O_WRONLY|O_CREAT|O_APPEND, 0644)) < 0 ) {
        if((log_fd=open("/dev/console", O_WRONLY|O_APPEND))<0) {
            fprintf(stderr, "Sorry, cannot open logfile and console\n");
            MUTEX_UNLOCK(&log_lock);
            return -1;
        }
        else {
            sprintf(line, "wcol: fail to open logfile-'%s'.\n", log_file);
            write(log_fd, line, strlen(line));
        }
    }

    /* month day time host program[pid]: message */
    sprintf(line, "%s %02d %02d:%02d:%02d %s %s[%d]: %s\n",
        mon_str[ntm->tm_mon], ntm->tm_mday,
        ntm->tm_hour, ntm->tm_min, ntm->tm_sec,
        host_name, log_name, XPID, rline);
    write(log_fd, line, strlen(line));
    close(log_fd);

    MUTEX_UNLOCK(&log_lock);

#endif /* USE_SYSLOG */

    return 0;
}

#ifndef LOG_INTERVAL
#define LOG_INTERVAL (32)
#endif /* LOG_INTERVAL */
#ifndef LOG_BUF_LEN
#define LOG_BUF_LEN (LOG_LINESIZE*(LOG_INTERVAL+2)+1)
#endif

static int icount=0;
static int ilen=0;
static char report_block[LOG_BUF_LEN];


/* you have to lock 'log_lock' */
void
LogSync()
{
    int nw;

#if 0
    fputs(report_block, stderr);
    nw = write(STDERR_FILENO, report_block, ilen);
#endif

    nw = write(fd_log, report_block, ilen);
    if(nw<0) {
    }

#if 0
    fprintf(stderr, "write %d bytes\n", nw);
#endif

    memset(report_block, 0, sizeof(report_block));
    icount = 0;
    ilen = 0;
}

/*
 * Log ---
 */
#ifdef __STDC__
int
Log(const char *fmt, ...)
#else
int
Log(va_alist)
va_dcl
#endif
{
    va_list args;
#ifndef __STDC__
    char *fmt;
#endif
    char tmp[LOG_LINESIZE];
    time_t now;
    register char *p, *q;
    char *r = &report_block[LOG_BUF_LEN-1];

#ifdef __STDC__
    va_start(args, fmt);
#else
    va_start(args);
    fmt = va_arg(args, char *);
#endif

    time(&now);

    MUTEX_LOCK(&log_lock);

    if(icount%LOG_INTERVAL==0) {
#if 0
        int nw;
        nw = write(fd_log, report_block, ilen);

        memset(report_block, 0, sizeof(report_block));
        icount = 0;
        ilen = 0;
#endif
        LogSync();
    }

    q = &report_block[ilen];
#if 0
    sprintf(tmp, "%d %s ", now, log_name);
#else
    sprintf(tmp, "| %d ", now);
#endif
    {
        p = tmp;
        while(*p && q<r) {
            *q++ = *p++;
            ilen++;
        }
        *q = '\0';
    }


    vsprintf(tmp, fmt, args);
    {
        p = tmp;
        while(*p && q<r) {
            *q++ = *p++;
            ilen++;
        }
        *q++ = '\n';
        ilen++;
        *q++ = '\0';
    }

    icount++;

    if(tmp[0]=='\0') {
        fprintf(stderr, "Log: tmp is empty\n");
    }


#ifdef LOG_REPORT_CONSOLE
    if(_trace_flag_) {
        fprintf(stderr,"=== LOG %d> %s\n", XPID, tmp);
        fflush(stderr);
    }
#endif

    MUTEX_UNLOCK(&log_lock);

out:
    va_end(args);

    return 0;
}

/*
 * LogLine ---
 */
int
LogLine(const char *tmp)
{
    register char *p, *q;
    char *r = &report_block[LOG_BUF_LEN-1];

    if(tmp==NULL || *tmp=='\0') {
        return -1;
    }

    MUTEX_LOCK(&log_lock);

    if(icount%LOG_INTERVAL==0) {
#if 0
        int nw;
        nw = write(fd_log, report_block, ilen);

        memset(report_block, 0, sizeof(report_block));
        icount = 0;
        ilen = 0;
#endif
        LogSync();
    }

    /* copy */
    q = &report_block[ilen];
    p = (char*) tmp;
    while(*p && q<r) {
        *q++ = *p++;
        ilen++;
    }
    *q++ = '\n';
    ilen++;
    *q++ = '\0';

    icount++;

#ifdef LOG_REPORT_CONSOLE
    if(_trace_flag_) {
        fprintf(stderr,"+++ %s\n", tmp);
        fflush(stderr);
    }
#endif

    MUTEX_UNLOCK(&log_lock);

    return 0;
}

/*
 * DLog ---
 */
#ifdef __STDC__
int
DLog(const char *fmt, ...)
#else
int
DLog(va_alist)
va_dcl
#endif
{
    va_list args;
#ifndef __STDC__
    char *fmt;
#endif
    char report[LOG_LINESIZE];
    struct timeval dnow;
    char *p;

#ifdef __STDC__
    va_start(args,fmt);
#else
    va_start(args);
    fmt = va_arg(args, char *);
#endif

    gettimeofday(&dnow, NULL);
    sprintf(report, "%d.%06d ", dnow.tv_sec, dnow.tv_usec);
    p = report;
    while(*p) {
        p++;
    }

    vsprintf(p, fmt, args);

    if(report[0]=='\0') {
        fprintf(stderr, "Log: report is empty\n");
    }
    else {
        RawLog(LOG_NOTICE, report);
    }

#ifdef LOG_REPORT_CONSOLE
    if(_trace_flag_) {
        fprintf(stderr,"=== LOG %d> %s\n", XPID, report);
        fflush(stderr);
    }
#endif

    va_end(args);

    return 0;
}



/*
 * Fatal ---
 */
#ifdef __STDC__
int
Fatal(const char *fmt, ...)
#else
int
Fatal(va_alist)
va_dcl
#endif
{
    va_list args;
#ifndef __STDC__
    char *fmt;
#endif
    char report[LOG_LINESIZE];
    char *p;

#ifdef __STDC__
    va_start(args,fmt);
#else
    va_start(args);
    fmt = va_arg(args, char *);
#endif

    /* append prefix */
    strcpy(report, "FATAL ");
    p = report + strlen(report);
    vsprintf(p, fmt, args);

    RawLog(LOG_WARNING, report);

    fprintf(stderr,"@@@ FATAL %d> %s\n", XPID, report);
    fflush(stderr);

    va_end(args);

    return 0;
}


int
SwitchLog(char *logpath, char *note)
{
    char logfname[LOG_FILENAMESIZE];
    int logfd;
    time_t ltime;
    time_t utime;


    /* get local and UTC date/time */
    time(&ltime);
    utime = mktime(gmtime(&ltime));

/*
    wcol_pid = getpid();
*/

    sprintf(logfname, "%s/log-%d", logpath, XPID);
    if((logfd = open(logfname, O_WRONLY|O_CREAT, 0644))<0) {
        Trace("SwitchLog: Cannot open new logfile-'%s'\n", logfname);
        return 1;
    }
    if(dup2(logfd, 2)<0) {
        Trace("SwitchLog: fail dup2() to stderr\n");
        return 1;
    }
#if 1
    if(dup2(logfd, 1)<0) {
        Trace("SwitchLog: fail dup2() to stdout\n");
        return 1;
    }
#endif


    /*
     * print information in head
     */
    if(note==NULL || *note == '\0') {
        Trace(";\n; log for wcol module\n;\n");
    }
    else {
        Trace(";\n; %s\n;\n", note);
    }
    Trace("; Pid %d, Ppid %d\n", getpid(), getppid());
    Trace("; Local %d : %s", ltime, ctime(&ltime));
    Trace("; UTC   %d : %s", utime, ctime(&utime));
    Trace(";\n");

    return 0;
}


int
#ifdef __STDC__
fdprintf(const int fd, const char *fmt, ...)
#else
fdprintf(va_alist)
va_dcl
#endif
{
    va_list args;
#ifndef __STDC__
    int fd;
    char *fmt;
#endif
    char buffer[LOG_LINESIZE];
    int nwrite;


#ifdef __STDC__
    va_start(args,fmt);
#else
    va_start(args);
    fd  = va_arg(args, int);
    fmt = va_arg(args, char *);
#endif

    vsprintf(buffer, fmt, args);
#if 0
    Trace("<<<%s", buffer);
#endif
try_write:
    nwrite = write(fd, buffer, strlen(buffer));
    if(nwrite<0) {
        if(errno==EINTR) {
            goto try_write;
        }
        else {
            Error("fdprintf: fail write() fd# %d (%d)", fd, errno);
        }
    }

    va_end(args);

    return nwrite;
}



