/*
 * simple stat routines
 *          by k-chinen@is.aist-nara.ac.jp, 1999
 *
 * $Id$
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>

#include <errno.h>

#include "sim_stat.h"
#include "msg.h"

int
sim_stat_clear(sim_rec *srec)
{
    int i;
    int chk;

    chk = MUTEX_LOCK(&srec->busy);

    for(i=0;i<srec->max;i++) {
        srec->hcount[i] = 0L;
    }
    srec->count = 0L;
    srec->igcount = 0L;

    chk = MUTEX_UNLOCK(&srec->busy);

    return 0;
}

int
sim_stat_alloc_new(sim_rec *addr, char *name, int sz, long *ar)
{
    sim_rec *ret;
    int chk;

    ret = addr;
    if(ret==NULL) {
        return NULL;
    }

    chk = pthread_mutex_init(&addr->busy, NULL);

    chk = MUTEX_LOCK(&addr->busy);
    strcpy(ret->name, name);
    ret->max = sz;
    ret->hcount = ar;
    chk = MUTEX_UNLOCK(&addr->busy);

    sim_stat_clear(ret);

    return 0;
}

sim_rec*
sim_stat_new(char *name, int sz)
{
    int chk;
    sim_rec *ret;

    ret = (sim_rec*) malloc(sizeof(sim_rec));
    if(ret==NULL) {
        return NULL;
    }

    chk = pthread_mutex_init(&ret->busy, NULL);

    chk = MUTEX_LOCK(&ret->busy);
    strcpy(ret->name, name);
    ret->max = sz;
    ret->hcount = (long*) malloc(sizeof(long)*sz);
    if(ret->hcount==NULL) {
        ret = NULL;
    }
    chk = MUTEX_UNLOCK(&ret->busy);

    sim_stat_clear(ret);

    return ret;
}




int
sim_stat_show(sim_rec *rec, FILE *fp)
{
    int i;
    int chk;
    long tsum, asum;

    chk = MUTEX_LOCK(&rec->busy);

    fprintf(fp, "=== %s ===\n", rec->name);

    if(rec->count<=0) {
        fprintf(fp, "no sample\n");
        chk = MUTEX_UNLOCK(&rec->busy);
        return -1;
    }


    fprintf(fp, "access %d time, sum %d -> ave %.3f\n",
        rec->count, rec->vsum, (double)rec->vsum/rec->count);

    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
    }
    tsum += rec->igcount;
    asum = tsum;

    fprintf(fp, "#%6s %9s %7s %9s %7s\n",
            "value", "count", "count%", "CDF", "CDF%");
    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
        if(rec->hcount[i]>0) {
            fprintf(fp, " %6d %9d %6.2f %9d %6.2f\n",
                i, rec->hcount[i], 100.0*(double)rec->hcount[i]/asum,
                tsum, 100.0*(double)tsum/asum);
        }
    }

    tsum += rec->igcount;
    fprintf(fp, " %6s %9d %6.2f %9d %6.2f\n",
        "others", rec->igcount, 100.0*(double)rec->igcount/asum,
         tsum, 100.0*(double)tsum/asum);


    fprintf(fp, "\n");

    chk = MUTEX_UNLOCK(&rec->busy);

    return 0;
}

int
sim_stat_show_lock(sim_rec *rec, FILE *fp)
{
    int i;
    int chk;
    long tsum, asum;


    fprintf(fp, "=== %s ===\n", rec->name);

    if(rec->count<=0) {
        fprintf(fp, "no sample\n");
        return -1;
    }


    fprintf(fp, "access %d time, sum %d -> ave %.3f\n",
        rec->count, rec->vsum, (double)rec->vsum/rec->count);

    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
    }
    tsum += rec->igcount;
    asum = tsum;

    fprintf(fp, "#%6s %9s %7s %9s %7s\n",
            "value", "count", "count%", "CDF", "CDF%");
    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
        if(rec->hcount[i]>0) {
            fprintf(fp, " %6d %9d %6.2f %9d %6.2f\n",
                i, rec->hcount[i], 100.0*(double)rec->hcount[i]/asum,
                tsum, 100.0*(double)tsum/asum);
        }
    }

    tsum += rec->igcount;
    fprintf(fp, " %6s %9d %6.2f %9d %6.2f\n",
        "others", rec->igcount, 100.0*(double)rec->igcount/asum,
         tsum, 100.0*(double)tsum/asum);


    fprintf(fp, "\n");

    return 0;
}

int
sim_stat_show_lock_fd(sim_rec *rec, int ofd)
{
    int i;
    int chk;
    long tsum, asum;


    fdprintf(ofd, "=== %s === (without lock)\n", rec->name);

    if(rec->count<=0) {
        fdprintf(ofd, "no sample\n");
        return -1;
    }


    fdprintf(ofd, "access %d time, sum %d -> ave %.3f\n",
        rec->count, rec->vsum, (double)rec->vsum/rec->count);

    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
    }
    tsum += rec->igcount;
    asum = tsum;

    fdprintf(ofd, "#%6s %9s %7s %9s %7s\n",
            "value", "count", "count%", "CDF", "CDF%");
    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
        if(rec->hcount[i]>0) {
            fdprintf(ofd, " %6d %9d %6.2f %9d %6.2f\n",
                i, rec->hcount[i], 100.0*(double)rec->hcount[i]/asum,
                tsum, 100.0*(double)tsum/asum);
        }
    }

    tsum += rec->igcount;
    fdprintf(ofd, "#%6s %9d %6.2f %9d %6.2f\n",
        "others", rec->igcount, 100.0*(double)rec->igcount/asum,
         tsum, 100.0*(double)tsum/asum);


    fdprintf(ofd, "\n");

    return 0;
}

int
sim_stat_show_HTML_nolock_fd(sim_rec *rec, int ofd)
{
    int i;
    int chk;
    long tsum, asum;
    int v;


    fdprintf(ofd, "<table border=\"0\" cellspacing=\"0\" cellpadding=\"2\">\n\
<tr bgcolor=\"#ff9999\"> <th colspan=\"7\">%s (without lock)\n", rec->name);


    if(rec->count<=0) {
        fdprintf(ofd, "<tr><th>no sample\n");
        goto out;
    }


    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
    }
    tsum += rec->igcount;
    asum = tsum;

    fdprintf(ofd, "<tr bgcolor=\"#ffcccc\">\n\
<th>%s <th>%s <th>%s <th>%s <th>%s <th width=\"390\">%s <th width=\"390\">%s\n",
            "value", "count", "PDF%", "sum", "CDF%",
            "graph of PDF", "graph of CDF");
    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
        if(rec->hcount[i]>0) {
            fdprintf(ofd,
"<tr><td align=\"right\"> %d \
<td align=\"right\"> %d \
<td align=\"right\"> %6.2f \
<td align=\"right\"> %d \
<td align=\"right\"> %6.2f  ",
                i, rec->hcount[i], 100.0*(double)rec->hcount[i]/asum,
                tsum, 100.0*(double)tsum/asum);

            v = (int)(100.0*(double)rec->hcount[i]/asum);
            if(v==100) {
                fdprintf(ofd,
"<td width=\"390\"><img width=\"200\" src=\"/img/bw200p100.jpg\"> 100  ");
            }
            else
            if(v<=0) {
                fdprintf(ofd,
"<td width=\"390\"> 0  ");
            }
            else
            if(v<=10 || v%10==0) {
                fdprintf(ofd,
"<td width=\"390\"><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                    v*2, v, v);
            }
            else {
                fdprintf(ofd,
"<td width=\"390\">\
<img width=\"%d\" src=\"/img/bw200p%02d.jpg\">\
<img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                    (v/10)*20, (v/10)*10, (v%10)*2, v%10 , v);
            }


            v = (int)(100.0*(double)tsum/asum);
            if(v==100) {
                fdprintf(ofd,
"<td width=\"390\"><img width=\"200\" src=\"/img/bw200p100.jpg\"> 100  ");
            }
            else
            if(v<=0) {
                fdprintf(ofd,
"<td width=\"390\"> 0  ");
            }
            else
            if(v<=10 || v%10==0) {
                fdprintf(ofd,
"<td width=\"390\"><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                    v*2, v, v);
            }
            else {
                fdprintf(ofd,
"<td width=\"390\"><img width=\"%d\" src=\"/img/bw200p%02d.jpg\">\
<img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                    (v/10)*20, (v/10)*10, (v%10)*2, v%10 , v);
            }

            fdprintf(ofd, "\n");
        }
    }

    if(rec->igcount>0) {
        tsum += rec->igcount;
        fdprintf(ofd,
"<tr><td align=\"center\"> %s\n\
<td align=\"right\"> %d\n\
<td align=\"right\"> %6.2f\n\
<td align=\"right\"> %d\n\
<td align=\"right\"> %6.2f\n",
            "others", rec->igcount, 100.0*(double)rec->igcount/asum,
             tsum, 100.0*(double)tsum/asum);

        v = (int)(100.0*(double)rec->igcount/asum);
        if(v==100) {
            fdprintf(ofd, "<td><img width=\"200\" src=\"/img/bw200p100.jpg\"> 100  ");
        }
        else
        if(v<=0) {
            fdprintf(ofd, "<td> 0  ");
        }
        else
        if(v<=10 || v%10==0) {
            fdprintf(ofd, "<td><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                v*2, v, v);
        }
        else {
            fdprintf(ofd,
"<td><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                (v/10)*20, (v/10)*10, (v%10)*2, v%10, v);
        }

        v = (int)(100.0*(double)tsum/asum);
        if(v==100) {
            fdprintf(ofd, "<td><img width=\"200\" src=\"/img/bw200p100.jpg\"> 100  ");
        }
        else
        if(v<=0) {
            fdprintf(ofd, "<td> 0  ");
        }
        else
        if(v<=10 || v%10==0) {
            fdprintf(ofd, "<td><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                v*2, v, v);
        }
        else {
            fdprintf(ofd,
"<td><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"><img width=\"%d\" src=\"/img/bw200p%02d.jpg\"> %d  ",
                (v/10)*20, (v/10)*10, (v%10)*2, v%10, v);
        }

        fdprintf(ofd, "\n");

    }


    fdprintf(ofd,
"<tr><th bgcolor=\"#ffcccc\">total <td align=\"right\"> %d\n\
<tr><th bgcolor=\"#ffcccc\">average <td align=\"right\"> %d\n\
<tr><th bgcolor=\"#ffcccc\">average <td align=\"right\"> %.1f\n",
        rec->count, rec->vsum, (double)rec->vsum/rec->count);

out:
    fdprintf(ofd, "</table>\n");

    return 0;
}

int
sim_stat_show_skip_lock_fd(sim_rec *rec, int ofd, int sk)
{
    int i;
    int chk;
    long tsum, asum;


    fdprintf(ofd, "=== %s === (with skip without lock)\n", rec->name);

    if(rec->count<=0) {
        fdprintf(ofd, "no sample\n");
        return -1;
    }


    fdprintf(ofd, "access %d time, sum %d -> ave %.3f\n",
        rec->count, rec->vsum, (double)rec->vsum/rec->count);

    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
    }
    tsum += rec->igcount;
    asum = tsum;

    fdprintf(ofd, "#%6s %9s %7s %9s %7s\n",
            "value", "count", "count%", "CDF", "CDF%");
    tsum = 0L;
    for(i=0;i<rec->max;i++) {
        tsum += rec->hcount[i];
#if 0
        if(rec->hcount[i]>0) {
            fdprintf(ofd, " %6d %9d %6.2f %9d %6.2f\n",
                i, rec->hcount[i], 100.0*(double)rec->hcount[i]/asum,
                tsum, 100.0*(double)tsum/asum);
        }
#else
        if(i%sk==0 && tsum>0) {
            fdprintf(ofd, " %6d %9d %6.2f %9d %6.2f\n",
                i, rec->hcount[i], 100.0*(double)rec->hcount[i]/asum,
                tsum, 100.0*(double)tsum/asum);
        }
#endif
    }

    tsum += rec->igcount;
    fdprintf(ofd, "#%6s %9d %6.2f %9d %6.2f\n",
        "others", rec->igcount, 100.0*(double)rec->igcount/asum,
         tsum, 100.0*(double)tsum/asum);


    fdprintf(ofd, "\n");

    return 0;
}

int
sim_stat_show_short(sim_rec *rec, FILE *fp)
{
    int chk;

    chk = MUTEX_LOCK(&rec->busy);
    fprintf(fp, "access %d time, sum %d -> ave %.3f\n",
        rec->count, rec->vsum, (double)rec->vsum/rec->count);
    chk = MUTEX_UNLOCK(&rec->busy);

    return 0;
}

int
sim_stat_reset(sim_rec *rec)
{
    int i;
    int chk;

    chk = MUTEX_LOCK(&rec->busy);

    rec->vsum = 0L;
    rec->count = 0L;
    for(i=0;i<rec->max;i++) {
        rec->hcount[i] = 0L;
    }
    rec->igcount = 0L;
    chk = MUTEX_UNLOCK(&rec->busy);

    return 0;
}

int
sim_stat_add_value(sim_rec *rec, long cv)
{
    int chk;

    chk = MUTEX_LOCK(&rec->busy);

    rec->count++;

    rec->vsum += cv;
    if(cv>=0 && cv< rec->max) {
        rec->hcount[cv]++;
    }
    else {
        rec->igcount++;
    }
    chk = MUTEX_UNLOCK(&rec->busy);

    return 0;
}

