/*
 * code-name 'KOTETU'
 *
 * squeue.c : main body for proxy server.
 *
 *      by k-chinen@is.aist-nara.ac.jp, 2000
 *
 * $Id$
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sched.h>
#include <pthread.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STROPTS_H
#include <stropts.h>
#endif
#ifdef HAVE_POLL_H
#include <poll.h>
#include <sys/poll.h>
#endif
#ifdef HAVE_SELECT_H
#include <sys/select.h>
#endif
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <strings.h>
#include <limits.h>
#include <sys/types.h>
#include <time.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <signal.h>
#include <setjmp.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <assert.h>
#include <errno.h>

/* private headers */
#include "wcol.h"
#include "socket.h"
#include "sigs.h"
#include "sim_stat.h"
#include "misc.h"
#include "http.h"
#include "cache.h"
#include "sess.h"
#include "squeue.h"
/*
#include "parse.h"
*/

#ifndef MAX
#define MAX(a,b) (((a)>(b))?(a):(b))
#endif

#ifdef TIMEVAL_TO_TIMESPEC
#define timeval_to_timespec(A,B) TIMEVAL_TO_TIMESPEC(A,B)	
#endif


int
sq_init(sess_queue_t *self, int sz)
{
    pthread_mutex_init(&self->lock,     NULL);
    pthread_cond_init(&self->not_empty, NULL);
    pthread_cond_init(&self->not_full,  NULL);

    self->cmax = sz;
    self->count = 0;

    self->head = (void*)NULL;
    self->tail = (void*)NULL;

    return 0;
}


int
sq_check(sess_queue_t *self)
{
    sess_t *cp;
    int cc;

    cc = 0;
    cp = (sess_t*) self->head;
    while(cp!=NULL) {
        if(cc>self->cmax) {
            Error("sq_check: overflow. there are %d cell(s) count %d/%d, sess %p",
                cc, self->count, self->cmax, self);

            return -1;
        }

        cp = cp->next;
        cc++;
    }

    goto ok;

fail:

    return -1;

ok:

    return 0;
}

int
sq_show(FILE *ofp, sess_queue_t *self)
{
    sess_t *cp;
    int cc;

    fprintf(ofp, "queue %8x: h %8x, t %8x, count %d/%d\n",
        self, self->head, self->tail, self->count, self->cmax);

    cc = 0;
    cp = (sess_t*) self->head;
    while(cp!=NULL) {
        fprintf(ofp, " %3d: %8x, n %8x, p %8x : #%d\n",
            cc, cp, cp->next, cp->prev, cp->sid);

        if(cc>self->cmax) {
            Error("sq_show: overflow. there are %d cell(s) count %d/%d, sess %p",
                cc, self->count, self->cmax, self);

            return -1;
        }

        cp = cp->next;
        cc++;
    }

    return 0;
}

int
sq_showaddr_mapfunc(FILE *ofp, sess_queue_t *self,
    void (func)(FILE*, void*, struct timeval*), struct timeval *ref)
{
    sess_t *cp;
    int cc;

    fprintf(ofp, "queue %8x: h %8x, t %8x, count %d/%d\n",
        self, self->head, self->tail, self->count, self->cmax);

    cc = 0;
    cp = (sess_t*) self->head;
    while(cp!=NULL) {
        fprintf(ofp, " %3d: %8x : #%d : ",
            cc, cp, cp->sid);
        fflush(ofp);
        func(ofp, (void*)cp, ref);

        if(cc>self->cmax) {
            Error("sq_showaddr_mapfunc: overflow. there are %d cell(s) count %d/%d, sess %p",
                cc, self->count, self->cmax, self);

            return -1;
        }

        cp = cp->next;
        cc++;
    }

    return 0;
}

int
sq_show_mapfunc(FILE *ofp, sess_queue_t *self,
    void (func)(FILE*, void*, struct timeval*), struct timeval *ref)
{
    sess_t *cp;
    int cc;
    int chk;
    char *msg;

    chk = pthread_mutex_trylock(&self->lock);
    if(chk) {
        if(chk==EBUSY) {
            msg = "BUSY";
        }
        else {
            msg = "ERR ";
        }
    }
    else {
        pthread_mutex_unlock(&self->lock);
        msg = "IDLE";
    }


    fprintf(ofp, "queue %8x: h %8x, t %8x, count %d/%d <%s>\n",
        self, self->head, self->tail, self->count, self->cmax, msg);

    cc = 0;
    cp = (sess_t*) self->head;
    while(cp!=NULL) {
        fprintf(ofp, " %3d: ", cc);
        fflush(ofp);

        if(cc>=self->cmax) {
            Error("sq_show_mapfunc: overflow. there are %d cell(s) count %d/%d, sess %p",
                cc, self->count, self->cmax, self);

            return -1;
        }

        func(ofp, (void*)cp, ref);

        cp = cp->next;
        cc++;
    }

    return 0;
}


int
sq_raw_push(sess_queue_t *self, sess_t *xcell)
{
#if 0
    fprintf(stderr, "sq_raw_push: cell %p\n", xcell);
#endif

    if(self->count>=self->cmax) {
        return -1;
    }

    xcell->prev = (sess_t*)self->tail;
    xcell->next = NULL;
    if(self->tail!=NULL) {
        ((sess_t*)self->tail)->next = (void*)xcell;
    }
    self->tail = (void*)xcell;
    if(self->head==NULL) {
        self->head = (void*)xcell;
    }

    self->count++;
    self->sig = (self->sig << 2) | ((int)xcell && 0xffff);

    return 0;
}

sess_t*
sq_raw_pull(sess_queue_t *self)
{
    sess_t *ret;
    int chk;

#if 0
    fprintf(stderr, "sq_raw_pull:\n");
#endif

    if(self->count<=0) {
        return NULL;
    }
#if 0
    chk = sq_check(self);
    if(chk) {
        chk = sq_show(stdout, self);
        return NULL;
    }
#endif
 assert(self->head!=NULL);
 assert(self->tail!=NULL);

    if(((sess_t*)self->head)->next != NULL) {
        ((sess_t*)self->head)->next->prev = NULL;
    }
    ret = (sess_t*)self->head;
    self->head = (void*)((sess_t*)self->head)->next;
    if(self->head==NULL) {
        self->tail = (void*)NULL;
    }

    self->count--;
    self->sig = (self->sig << 2) | ((int)ret && 0xffff);

    return ret;
}



int
sq_push(sess_queue_t *self, sess_t *xcell)
{
    int chk;

#if 0
    fprintf(stderr, "sq_push: self %p, cell %p\n", self, xcell);
#endif

    if(self==NULL || xcell==NULL) {
        Error("why do you want to push empty cell");
        return -1;
    }

    pthread_mutex_lock(&self->lock);


    while(self->count == self->cmax) {
        chk = pthread_cond_wait(&self->not_full, &self->lock);
        if(chk==ETIMEDOUT) {
            chk = -1;
            goto out;
        }
    }

#if 0
    sq_show(stderr, self);
#endif

    chk = sq_raw_push(self, xcell);
    if(!chk) {
        pthread_cond_signal(&self->not_empty);
    }

#if 0
    sq_show(stderr, self);
#endif

out:
    pthread_mutex_unlock(&self->lock);

    return chk;
}


int
sq_timedpush(sess_queue_t *self, sess_t *xcell, int wait_time)
{
    int chk;
    struct timespec join_time;

#if 0
    fprintf(stderr, "sq_timedpush: self %p, cell %p\n", self, xcell);
#endif

    if(self==NULL || xcell==NULL) {
        Error("why do you want to push empty cell");
        return -1;
    }

    pthread_mutex_lock(&self->lock);

#ifdef CLOCK_REALTIME
    clock_gettime(CLOCK_REALTIME, &join_time);
#else
	{
	struct timeval tvtime;
	gettimeofday(&tvtime, NULL);
	timeval_to_timespec(&tvtime, &join_time);
	}
#endif
    join_time.tv_nsec   += wait_time;
    join_time.tv_sec    += (join_time.tv_nsec/1000000000);
    join_time.tv_nsec   = (join_time.tv_nsec%1000000000);


    while(self->count == self->cmax) {
        chk = pthread_cond_timedwait(&self->not_full, &self->lock, &join_time);
        if(chk==ETIMEDOUT) {
            chk = -1;
            goto out;
        }
    }

#if 0
    sq_show(stderr, self);
#endif

    chk = sq_raw_push(self, xcell);
    if(!chk) {
        pthread_cond_signal(&self->not_empty);
    }

#if 0
    sq_show(stderr, self);
#endif

out:
    pthread_mutex_unlock(&self->lock);

    return chk;
}


sess_t*
sq_pull(sess_queue_t *self)
{
    sess_t *cur;
    int chk;

#if 0
    fprintf(stderr, "sq_pull: self %p\n", self);
#endif

    pthread_mutex_lock(&self->lock);

    while(self->count==0) {
        chk = pthread_cond_wait(&self->not_empty, &self->lock);
        if(chk==ETIMEDOUT) {
            cur = NULL;
            goto out;
        }
    }

    /* main */
    cur = sq_raw_pull(self);
    if(cur!=NULL) {
        pthread_cond_signal(&self->not_full);
    }
#if 0
    sq_show(stderr, self);
#endif

out:
    pthread_mutex_unlock(&self->lock);

    return cur;
}

sess_t*
sq_pull_nonblock(sess_queue_t *self)
{
    sess_t *cur;
    int chk;

#if 0
    fprintf(stderr, "sq_pull_nonblock: self %p\n", self);
#endif

    pthread_mutex_lock(&self->lock);

    if(self->count==0) {
        cur = NULL;
        goto out;
    }

    /* main */
    cur = sq_raw_pull(self);
    if(cur!=NULL) {
        pthread_cond_signal(&self->not_full);
    }
#if 0
    sq_show(stderr, self);
#endif

out:
    pthread_mutex_unlock(&self->lock);

    return cur;
}

int
sq_pullchunk(sess_queue_t *self, sess_t *xchunk[], int range)
{
    sess_t *cur;
    sess_t **pos;
    int chk;
    int count;

#if 0
    fprintf(stderr, "sq_pullchunk: self %p, chunk %p [%d]\n",
        self, xchunk, range);
#endif

    count = 0;
    pos = xchunk;

    chk = pthread_mutex_lock(&self->lock);
    if(chk) {
        fprintf(stderr, "sq_pullchunk: self %p, fail lock (%d)\n", self, chk);
    goto out;
    }

    while(self->count==0) {
        chk = pthread_cond_wait(&self->not_empty, &self->lock);
        if(chk==ETIMEDOUT) {
            cur = NULL;
            goto done;
        }
    }

    do {
    cur = sq_raw_pull(self);
        if(cur!=NULL) {
        *pos++ = cur;
        count++;
    }
    } while(self->count>0 && count<range);

    if(cur!=NULL) {
        pthread_cond_signal(&self->not_full);
    }

done:
    pthread_mutex_unlock(&self->lock);

out:
#if 0
 {
 int i;
    if(count>0) {
        Trace("sq_pullchunk: count %d/range %d\n", count, range);
        for(i=0;i<count;i++) {
            Trace(" %4d: %p : sess# %d\n", i, xchunk[i], xchunk[i]->sid);
        }
    }
    else {
        Trace("sq_pullchunk: nocount\n");
    }
 }
#endif

    return count;
}


sess_t*
sq_timedpull(sess_queue_t *self, int wait_time)
{
    sess_t *cur;
    struct timespec join_time;
    int chk;
    int err;

#if 0
    fprintf(stderr, "sq_timedpull: self %p, wait %d\n", self, wait_time);
#endif

#ifdef CLOCK_REALTIME
    clock_gettime(CLOCK_REALTIME, &join_time);
#else
	{
	struct timeval tvtime;
	gettimeofday(&tvtime, NULL);
	timeval_to_timespec(&tvtime, &join_time);
	}
#endif
    join_time.tv_sec    += wait_time;

    pthread_mutex_lock(&self->lock);

    while(self->count==0) {
        chk = pthread_cond_timedwait(&self->not_empty, &self->lock, &join_time);
        err = errno;
        if(chk) {
#if 0
            fprintf(stderr, "errno %d\n", errno);
            fprintf(stderr, "err   %d\n", err);
#endif
            if(errno==ETIMEDOUT) {
#if 0
                fprintf(stderr, "*** timeout ***\n");
#endif
            }
            cur = NULL;
            goto out;
        }

    }

    /* main */
    cur = sq_raw_pull(self);
    if(cur!=NULL) {
        pthread_cond_signal(&self->not_full);
    }
#if 0
    sq_show(stderr, self);
#endif

out:
    pthread_mutex_unlock(&self->lock);

    return cur;
}



extern pthread_mutex_t sess_lock;
extern size_t  sess_size;

static sess_t *sq_storage=NULL;
static sess_t *sq_freelist=NULL;

void
sq_show_freelist()
{
    sess_t *p;
    int i;

    p = sq_freelist;
    i = 0;
    while(p) {
        Trace("%5d: %p %p\n", i, p, p->next);

        p = p->next;
        i++;
    }
}

off_t
sq_alloc_list(int n)
{
    int i;
    off_t listsize;

    listsize = n*sizeof(sess_t);
    Trace("sq_alloc_list: %d elements (size %d), %lld bytes(%ldM+%ld)\n",
        n, sizeof(sess_t),
        (long long)listsize, (long)dM(listsize), (long)mM(listsize) );

    MUTEX_LOCK(&sess_lock);

    sq_storage = (sess_t*)MALLOC(listsize);
    if(sq_storage==NULL) {
        Error("not enough memory for free sesssion queue");
        abort();
    }

    for(i=0;i<n-1;i++) {
        sq_storage[i].prev = NULL;
        sq_storage[i].next = &sq_storage[i+1];
    }
    sq_storage[n-1].prev = NULL;
    sq_storage[n-1].next = NULL;

    sq_freelist = &sq_storage[0];

#if 0
    sq_show_freelist();
#endif

    MUTEX_UNLOCK(&sess_lock);

    return listsize;
}

sess_t*
sq_get_free()
{
    sess_t *r;
    sess_t *n;

    MUTEX_LOCK(&sess_lock);

    if(sq_freelist==NULL) {
        Error("no more free sesssion");
        r = NULL;
        goto done;
    }

    r = sq_freelist;
    n = sq_freelist->next;
    sq_freelist = sq_freelist->next;
    r->next = NULL;

done:
    MUTEX_UNLOCK(&sess_lock);

    return r;
}


int
sq_be_free(sess_t*x)
{

    if(x==NULL) {
        return -1;
    }

    MUTEX_LOCK(&sess_lock);

    x->next = sq_freelist;
    x->prev = NULL;
    sq_freelist = x;

    MUTEX_UNLOCK(&sess_lock);

    return 0;
}




#ifdef SQUEUE_TEST

main()
{
    sess_queue_t a;
    sess_queue_t *ap;
    sess_t *c;
    void *m;
    int achk;
    void *schk;


    m = (void*)malloc(BUFSIZ);
    c = (sess_t*)malloc(sizeof(sess_t));

    sq_init(&a, 10);
    ap = &a;

    achk = sq_push(ap, c);      if(achk)    Error("a line %d", __LINE__);
    achk = sq_show(stderr, ap);
#if 0
    achk = sq_push(ap, m);      if(achk)    Error("a line %d", __LINE__);
    achk = sq_push(ap, m);      if(achk)    Error("a line %d", __LINE__);
    achk = sq_push(ap, m);      if(achk)    Error("a line %d", __LINE__);
    schk = sq_pull(ap);         if(!schk)   Error("s line %d", __LINE__);

#endif


}

#endif /* SQUEUE_TEST */

