open Ls
open Format

(* ---------------------------------------- *)
(***** interface for command line tools *****)
(* ---------------------------------------- *)
type command 'a = (string -> 'a) -> string * out_channel -> string list -> 'a

(* measure executing time *)
let chronicle f x =
  let time = Unix.gettimeofday () in
  let fx = f x in
  fx, (Unix.gettimeofday () -. time)

let watcher description f x =
  let fx, time = chronicle f x in
  if !Ref.show_time then
    begin
      Format.eprintf "%s\t: %.5fs@." description time;
      fx
    end
  else
    fx

let watcher_l description f x =
  watcher (description^"\n\t") f x

let write_file out_ch =
  watcher "writing\t"
    (List.iter (fun line ->
      fprintf (formatter_of_out_channel out_ch) "%s@." line))

(* IO functions *)
(* ---------------------------------------- *)
(* g is closer *)
let finally x f g =
  try 
    let y = f x in g x; y
  with
    e -> g x; raise e

(* read string list from stdin *)
let rec read_lines in_ch =
  try
    let line = input_line in_ch in
    line :: read_lines in_ch
  with End_of_file ->
    close_in in_ch;
    []

(* join by '\n'; read string from stdin *)
let read_all in_ch = String.concat "\n" (read_lines in_ch)

let close_temp_file (tmp, ch) =
  close_out ch ;
  Sys.remove tmp

(* exe CMD on CLI -> read all stdout (string) *)
let run command : string =
  let run' () =
    finally
      (Unix.open_process_in command)
      (fun ch -> read_all ch)
      (fun ch -> ignore (Unix.close_process_in ch))
  in watcher_l ("running\t"^command) run' ()

(* difference of run is "string list" *)
let run_lines command : string list =
  finally
    (Unix.open_process_in command)
    (fun ch -> read_lines ch)
    (fun ch -> ignore (Unix.close_process_in ch))


(* run with file *)
let run_with_body
    (closing   : string * out_channel -> unit)
    (f_command : string -> 'a)
    (path      : string * out_channel)
    (lines     : string list)
    = let exec (f, out_ch) =
      write_file out_ch lines; f_command f in
    finally path exec closing

(* run IO function with file -> read all stdout *)
let run_with f_command =
  run_with_body close_temp_file f_command

(* not remove file *)
let run_with_save f_command =
  run_with_body (fun (_,ch) -> close_out ch) f_command

(** SAT FUNCTIONS **)
(* --------------------------------------------- *)
(** SAT using Yices **)
let yices () = "yices -e "

open Lexing
let syntax_error path p =
  eprintf "Yices file %S at line %d, character %d:@.Syntax error.@."
    path p.pos_lnum (p.pos_cnum - p.pos_bol)
  ; exit 1

(* return evidence if sat *)
let sat_ run_func (path,ch) lines =
  let parse x =
    let lexbuf = Lexing.from_string x in
    try
      Yices_parser.sat Yices_lexer.evidence lexbuf
    with Parsing.Parse_error ->
      syntax_error path lexbuf.lex_curr_p
  in (* RUN! *)
  run_func
    (fun tmp ->
      parse (run (yices () ^ tmp)))
    (path,ch)
    lines

(* save as temporary file *)
let sat_save lines = sat_ run_with_save (Filename.open_temp_file "yices" ".ys") lines

(* create and open temp file with tmp dir *)
let open_temp_here tmp pre suf =
  (try Unix.mkdir tmp 0o775 with Unix.Unix_error (_,_,_) -> ());
  Filename.open_temp_file ~temp_dir:tmp pre suf

(* use non removing version if output path set *)
let sat lines = match !Ref.dump with
  | Some temp_dir ->
      sat_ run_with_save (open_temp_here temp_dir "yices" ".ys") lines
  | None      ->
      sat_ run_with      (Filename.open_temp_file "yices" ".ys") lines

let is_sat lines =
  match sat lines with
  | Some _ -> true
  | None   -> false


(** MAXSAT **)

let maxsat_ run_func (path,ch) lines =
  let parse x =
    let lexbuf = Lexing.from_string x in
    try
      Yices_parser.max_sat Yices_lexer.evidence lexbuf
    with Parsing.Parse_error ->
      syntax_error path lexbuf.lex_curr_p
  in (* RUN! *)
  run_func
    (fun tmp ->
      parse (run ("yices -e " ^ tmp)))
    (path,ch)
    lines

let maxsat lines = match !Ref.dump with
  | Some temp_dir ->
      maxsat_ run_with_save (open_temp_here temp_dir "yices" ".ys") lines
  | None      ->
      maxsat_ run_with      (Filename.open_temp_file "yices" ".ys") lines

let is_maxsat lines =
  match maxsat lines with
  | Some _ -> true
  | None   -> false

(** CHECK **)
let sat_check () = is_sat ["(assert true)(check)"]

let max_check () = is_maxsat ["(assert+ true)(check)"]
