type f_type =
  | BOOL | NAT | INT | REAL | BITVECTOR of int

type 'a term =
  | A of string (* atomic term *)
  | V of string * f_type (* name and type *)
  | BV of bool list (* bit-vector *)
  | Tuple of ('a form) list

(* define basic boolean operators *)
and 'a form =
  | True
  | False
  | Term of ('a term)
  | N of ('a form)
  | C of ('a form) list
  | D of ('a form) list
  | IMP of ('a form) * ('a form)
  | EQ of ('a form) * ('a form)
  | GT of ('a form) * ('a form)
  | LT of ('a form) * ('a form)
  | OP of string * ('a form list) (* direct assertion *)

type f = string form

let name = function
  | Term (V (n,_)) -> n
  | _              -> invalid_arg "Form.name: no variable"

(* alias *)
let atom s    = Term (A s)
let var v t   = Term (V (v,t))
let bv bs     = Term (BV bs)
let tuple fs  = Term (Tuple fs)

let neq (a,b)  = N ( EQ (a,b) )
let geq (a,b)  = D [ EQ (a,b); GT (a,b) ]
let leq (a,b)  = D [ EQ (a,b); LT (a,b) ]

(* combinator for forms *)
let (==>%) x y = IMP (x,y)
let (=%)   x y = EQ  (x,y)
let (<>%)  x y = neq (x,y)
let (>%)   x y = GT  (x,y)
let (<%)   x y = LT  (x,y)
let (>=%)  x y = geq (x,y)
let (<=%)  x y = leq (x,y)

let to_TF p x y = if p x y then True else False
(* combinator for boolean *)
let (=?)  x y = to_TF (=)  x y
let (<>?) x y = to_TF (<>) x y
let (>?)  x y = to_TF (>)  x y
let (<?)  x y = to_TF (<)  x y
let (>=?) x y = to_TF (>=) x y
let (<=?) x y = to_TF (<=) x y
let (&&?) x y = to_TF (&&) x y
let (||?) x y = to_TF (||) x y

(* formatting function *)
let rec simply = function
  | C []         -> True
  | D []         -> False
  | C [f]        -> f
  | D [f]        -> f
  | C cnf        -> C (List.map simply cnf)
  | D dnf        -> D (List.map simply dnf)
  | OP (bin,fs)  -> OP (bin,(List.map simply fs))
  | IMP (x,y)    ->
      begin
        match simply x, simply y with
        | False,_     -> True
        | _,True      -> True
        | True,False  -> False
        | _           -> IMP (x,y)
      end
  | EQ (x,y)  -> EQ (simply x, simply y)
  | GT (x,y)  -> GT (simply x, simply y)
  | LT (x,y)  -> LT (simply x, simply y)
  | Term t    -> Term
    begin
      match t with
      | Tuple fs -> Tuple (List.map simply fs)
      | atom     -> atom
    end
  | other     -> other


let rec variables = function
  | N f          -> variables f
  | C cnf        -> Ls.uniq [v | f <- cnf; v <- variables f]
  | D dnf        -> Ls.uniq [v | f <- dnf; v <- variables f]
  | OP (bin,fs)  -> Ls.uniq [v | f <- fs; v <- variables f]
  | IMP (a,b)    -> Ls.uniq @@ variables a @ variables b
  | EQ (a,b)     -> Ls.uniq @@ variables a @ variables b
  | GT (a,b)     -> Ls.uniq @@ variables a @ variables b
  | LT (a,b)     -> Ls.uniq @@ variables a @ variables b
  | Term t       -> begin
      match t with
      | V _      -> [t]
      | Tuple fs -> [v | f <- fs; v <- variables f]
      | _        -> [] end
  | _          -> []


(* max sat *)
type 'a max_form =
  | MAX of (int * 'a form) (* with weight *)
  | NMX of ('a form) (* non maximal form *)

let max_to_form = function
  | MAX (_,f) -> f
  | NMX f     -> f
