open Ls
open Term
open Result

open Format
open Arg

let usage = "\
Webpage: www.jaist.ac.jp/~s1310032/coll or www.jaist.ac.jp/~hirokawa/

CoLL is confluence (commutation) prover based on
Hindley's commutation theorem for left-linear TRS.

Usage: coll [options] trs1 [trs2 ...]
  1. prove confluence of TRS R
    => coll [options] R
  2. prove commutation of TRSs R, S .. and T
    => coll [options] R S .. T"

(*** this is Main module ***)
let comm_checker r s =
  let rr,ss = Trs.sort r, Trs.sort s in
  if !Ref.no_comm  then
    if !Ref.tcap then
      match Commute.noncomm rr ss with
      | NO nf -> NO nf
      | other -> Commute.direct rr ss
    else
      Commute.direct rr ss
  else
    Commute.commute rr ss

let cr_checker r = comm_checker r r

(** for composable decomposition **)

let if_yes rss proofs =
  printf "YES\n@.";
  if !Ref.no_proof then
    ()
  else
    begin
      if !Ref.no_decompose then
        Trs.print_rules (union rss)
      else
        Compose.show_decompose rss;
        List.iter Commute.show_proof proofs
    end

let if_no rss (t,s,u) =
  printf "NO\n@.";
  if !Ref.no_proof then
    ()
  else
    begin
      if !Ref.no_decompose then
        Trs.print_rules (union rss)
      else
        Compose.show_decompose rss;
        printf "unjoinable peak@.";
        printf "%4s%s\n*<- %s ->*\n%4s%s@."
          "" (sprint t) (sprint s) "" (sprint u);
    end

let if_maybe rs =
  printf "MAYBE\n@."
  (*
  if_no_proof @@
    Compose.print_decomps rs;
    printf "commutation of following TRSs cannot be solved@.";
    printf "--- R\n%s\n--- S\n%s@."
    (Trs.sprint_rules r)
    (Trs.sprint_rules s)
   *)

(* This is miso for secret (hint 60.trs) *)
let reduce rs = if !Ref.no_red then rs else Step.reduce !Ref.k rs

let run rs ss =
  if not !Ref.no_ll && not (Trs.left_linear (rs@ss)) then
    print_endline "MAYBE\ninput TRSs are not left-linear"
  else
    let r,s = reduce rs, reduce ss in
    match comm_checker r s with
    | YES proof -> if_yes [r;s] [proof]; Step.if_reduced rs r; Step.if_reduced ss s;
    | NO nfp    -> if_no  [r;s] nfp    ; Step.if_reduced rs r; Step.if_reduced ss s;
    | MAYBE     -> if_maybe r

let run2 rs =
  if not !Ref.no_ll && not (Trs.left_linear rs) then
    print_endline "MAYBE\ninput TRSs are not left-linear"
  else
    let r = reduce rs in
    let modular =
      if !Ref.no_decompose then [r] else Compose.decomposeCOM r
      (* if !Ref.no_decompose then [r] else Compose.cons_share r *)
    in match Result.sequence_with cr_checker modular with
    | YES proofs -> if_yes modular proofs; Step.if_reduced rs r;
    | NO nfp     -> if_no  modular nfp   ; Step.if_reduced rs r;
    | MAYBE      -> if_maybe r

(* ------------------------------------------------------ *)
(*
  System Architecture: consider input TRSs
   - TRS           -> | TRS    : CR      |             -> Result
   - TRS1,TRS2     -> | TRS1,2 : Commute |             -> Result
   - TRS^n (n > 2) -> | prove Commutation each other | -> Results
*)
(* ------------------------------------------------------ *)
(* tank of input TRSs *)
let rss = ref ([] : trs list)

(* if input TRSs are at least two TRSs *)
let rec comm_mode = function
  | []        -> []
  | rs :: rss -> [rs, ss | ss <- rss] @ comm_mode rss

(* check all dependent tools *)
let env_test () =
  let succ s  = printf "%s@." s
  and error s = printf "%s@." s; exit 1 in
  let trying f x s e =
    try
      begin ignore (f x); succ s end
    with _ -> error e in
  if not !Ref.test
  then
    ()
  else
    begin
      printf "Testing...@.";
      trying Console.sat_check () "Yices 1 exists" "Yices 1 does not exist";
      printf "Done@.";
      exit 0
    end


(***************************************************************)
(*********************** OPTION SETTINGS ***********************)
let item s = "", Unit (fun () -> ()), "\n"^s

let functions = [
   item "Functions:";
   ("--no-proof"     , Set Ref.no_proof, "" );
   ("-np"            , Set Ref.no_proof,
       " no proof (long is --no-proof)");
   ("--no-comm"      , Set Ref.no_comm, "" );
   ("-nc"            , Set Ref.no_comm,
       " disable commutative decomposition to apply only single theorem (long is --no-comm)");
   ("--no-decompose" , Set Ref.no_decompose, "");
   ("-nd"            , Set Ref.no_decompose,
       " disable decomposition by modularity (long is --no-decompose)");
   ("--no-ll"        , Set Ref.no_ll        , "");
   ("-nll"           , Set Ref.no_ll,
       " allow non-left-linear TRS (resulting proof can be wrong) (long is --no-ll)");
   ("--no-red"       , Set Ref.no_red        , "");
   ("-nr"           , Set Ref.no_red        ,
       " disable rule elimination (long is --no-red)");
   ("-k"             , Set_int Ref.k        ,
       "<x>" ^ sprintf " use at most x steps for rewriting (default %d)" !Ref.k);
]

let thm_specs = [
  item "Theorem-Flags: if some blow flags given, other flags are disabled";
  ("-dc" , Set Ref.dc , " use development closedness");
  (* ("-sc" , Set Ref.sc , " use strict commutation (Geser 90)"); *)
  ("-rl" , Set Ref.rl , " use rule labeling");
  (* jk family *)
  ("-kb" , Set Ref.kb , " use Knuth and Bendix");
  ("-jk" , Set Ref.jk , " use Jouannaud and Kirchner (set E = {A,C,AC})");
  ("-jka"   , Set Ref.jka   , " set E = {A}");
  ("-jkc"   , Set Ref.jkc   , " set E = {C}");
  ("-jkac"  , Set Ref.jkac  , " set E = {AC}");
  ("-tcap"  , Set Ref.tcap  , " use TCAP approximation (unjoinability checking) ")
]

let debug_specs = [
  item "Debug-Options:";
  ("--debug"     , Set Ref.debug       , " show newer commuting pair for every cycle");
  ("--test"      , Set Ref.test        , " test your computer environment");
  ("--show-time" , Set Ref.show_time   , " show each processing time");
  ("--dump"      , String Ref.set_dump , "<path> save Yices files in path-dir");
  ("--off"       , Clear Ref.flag , "" (* hidden option *))
]


let () = parse
  ( align @@
      functions
    @ thm_specs
    @ debug_specs
    @ [item "Others:"] (* vertical skip (split to Debug and help) *)
  )
  (* accumulate TRS *)
  (fun rs -> rss := !rss @ [Read.read_trs rs])
  usage
  ; (* end of Arg.parse *)
  env_test ();
  Step.cache_on (); Join.cache_on (); Commute.cache_on ();
  Ref.flag_on ();
  (* Ref.flag_off () *)
  match !rss with
  | []   -> eprintf "ERROR\nno TRSs@."
  | [rs] -> run2 (Trs.init rs)
  | rss  -> List.iter (fun (r,s) -> run r s)
    [ Trs.init r, Trs.init s | r,s <- comm_mode rss ]
