module type OrderedType = sig 
  type t 
  val compare : t -> t -> int
end

module type S = sig
  type elt
  val index : elt -> int
  val element : int -> elt 
  val refresh : unit -> unit
end

let rec inv_assoc idx = function 
  | []      -> raise Not_found
  | (elt, idx') :: alist when idx = idx' -> elt
  | _ :: alist -> inv_assoc idx alist

module Int = struct 
  type t = int
  let compare = compare
end

module IntMap = Map.Make (Int)

module Make (Elt : OrderedType) = struct

  type elt = Elt.t

  module EltMap = Map.Make (Elt)

  let table = ref (EltMap.empty, IntMap.empty)
  
  let count = ref 0 
  
  let new_index () = incr count; !count
  
  let index elt = 
    let elt_map, int_map = !table in
    try
      EltMap.find elt elt_map 
    with
      Not_found ->
        let idx = new_index () in
        table := (EltMap.add elt idx elt_map, 
                  IntMap.add idx elt int_map);
        idx

  let element idx = IntMap.find idx (snd !table)

  let refresh () =
    count := 0;
    table := (EltMap.empty, IntMap.empty)
      
end
