open Term

let rec remove_first elem = function
  | [] -> []
  | h :: t -> if elem = h then t else h :: (remove_first elem t)

let remove_common_args l1 l2 =
  let rec rca1 ll acc = function
    | [] -> ll, List.rev acc
    | h :: t -> 
	if List.mem h ll then
          let lln = remove_first h ll in rca1 lln acc t
        else rca1 ll (h::acc) t
  in
  rca1 l1 [] l2
    
let group s t =
 match s,t with
 | F(f,ss), F(g,tt) when f = g ->
     let ss1,tt1 = remove_common_args ss tt in
     f, Ls.count ss1, Ls.count tt1
 | _ -> failwith "invalid arg in Ac.normalize"

let head f = function
  | F(g,gs) when f=g -> true
  | _ -> false

let rec flatten ac = function
  | V _ as v -> v
  | F(f,ts) ->
      let ts_flat = List.map (flatten ac) ts in 
      if not (List.mem f ac) then F(f, ts_flat)
      else
	if List.exists (head f) ts 
	then flatten ac (F(f, iter f ts_flat))
	else F(f,ts_flat)
and iter f = function
  | [] -> []
  | (F(g,gs)) :: tt when f=g -> gs @ (iter f tt)
  | h :: tt -> h :: (iter f tt)

(* solve Diophantine equations; x+x+z = y+y ==> 2x+z = 2y *)
let basic_sol ac h1 h2 = 
  let f, ss, tt = group (flatten ac h1) (flatten ac h2) in
  let bb1,bb2 = (List.map snd ss), (List.map snd tt) in
  let aa = List.map fst (ss @ tt) in 
  let bs = List.map (fun (x,y) -> x @ y) (Dio.basis bb1 bb2) in 
  (f, aa, bs)

(* assume s and t has are elementary *)
let basic_sol_proto ac s t =
  let f, ss, tt = group (flatten ac s) (flatten ac t) in
  let terms_s = List.map fst ss
  and terms_t = List.map fst tt in
  let lhs, rhs = [ i | _, i <- ss ], [ j | _, j <- tt ] in
  (* terms_s, terms_t, Dio.basis_proto lhs rhs *)
  terms_s, terms_t, Dio.basis lhs rhs
  
let rec make_ac_term f v = function
    0 -> None
  | 1 -> Some v
  | n -> Some (make_ac_term_aux f v n)
and make_ac_term_aux f v = function
  | n when n = 2 -> F (f, [v; v])
  | n when n > 2 ->
     let t = make_ac_term_aux f v (n-1) in
     F (f, [v; t])
  | _ -> invalid_arg "make_ac_term"

let rec make_ac_term_from (f,one) = function
    [] -> one
  | [v] -> v
  | [v1;v2] -> F (f, [v1;v2])
  | v1::vs -> F (f, [v1; make_ac_term_from (f,one) vs])

let merge vec vecs = List.map2 (fun xs xss -> xs::xss) vec vecs

(* vec :: string list; sols :: int list list *)
let build_ac1_unifier (f,one) (vec, gens) =
  let rec loop f (vec, gens) acc =
    match gens with
      [] -> acc
    | gen::gens -> 
       let new_v = Term.new_var () in
       loop f (vec, gens)
         (merge
            [make_ac_term f new_v i | v,i <- List.combine vec gen]
            acc)
  in
  let var_map = List.combine
                  vec
                  (loop f (vec, gens) (Ls.rep (List.length vec) []))
  in
  [x, make_ac_term_from (f,one) [v | Some v <- gen]
  | (V x),gen <- var_map ]

(* assume |ac| = 1 *)
let elementary_ac1_unify ac one s t =
  let s' = flatten ac s in
  let t' = flatten ac t in
  match (s', t') with
    F (f, ss), F (g,tt) when f = g ->
     (*let vsl, vsr, sols = basic_sol_proto ac s t in
     build_ac1_unifier (f,one) (vsl@vsr, [soll@solr | soll, solr <- sols])
      *)
     let _, vs, sols = basic_sol ac s t in
     build_ac1_unifier (f,one) (vs, sols)
  | V x, _ -> [x, t]
  | _, V x -> [x, s]
  | _ -> []
  
let rec erase (ac,one) = function
    V x -> V x
  | t when t = one -> one
  | F (f, [t1;t2]) when List.mem f ac ->
     let t1' = erase (ac,one) t1
     and t2' = erase (ac,one) t2
     in
     if t1' = one then t2'
     else if t2' = one then t1'
     else F (f, [t1';t2'])
  | F (f, ts) -> F (f, [erase (ac,one) t | t <- ts])

let elementary_ac_unify ac s t =
  let one = new_const () in
  let ac1_mgu = elementary_ac1_unify ac one s t in
  let var_range = Ls.uniq [ v | _,t <- ac1_mgu; v <- vars t ] in
  let admissible_sigmas = [ [v,one | v <- vs] | vs <- Ls.powset var_range ] in
  let mksubst sigma = [x, erase (ac,one) v'
                  | x,v <- ac1_mgu; v' <- [subst sigma v]] in
  [subst | sigma <- admissible_sigmas; subst <- [mksubst sigma];
         List.for_all (fun (_,v) -> v <> one) subst]


let filterPreCond4 basic_sols aa = 
  let i1n = Ls.range 0 ((List.length aa)-1) in
  let violates4 b i = (List.nth b i >= 2) && (not (is_var (List.nth aa i))) in
  [ b | b <- basic_sols; not (List.exists (violates4 b) i1n) ]

(*
let bs1 = filterPreCond4 b aa
*)

let condition3 base aa =
 let base_ar = Array.of_list (List.map Array.of_list base) in
 let b = ref true in 
 for i=0 to (List.length aa) - 1 do
  let column_i = [ base_ar.(j).(i) | j <- Ls.range 0 ((Array.length base_ar) - 1) ] in
  begin
  if List.for_all (fun x -> x=0) column_i 
  then b := false  
  end;
 done; !b

(*
let bss3 = [ bs | bs <- Listx.power bs2; bs <> [] ]
let bss4 = [ bs | bs <- bss3; condition3 bs aa ]
*)


let condition4 base aa =
  let base_ar = Array.of_list (List.map Array.of_list base) in
  let b = ref true in 
  for i=0 to (List.length aa) - 1 do
    if not (is_var (List.nth aa i)) then
      let column_i = [ base_ar.(j).(i) | j <- Ls.range 0 ((Array.length base_ar) - 1) ] in
      let sum = List.fold_left (+) 0 column_i in
      begin
	if not (sum = 1) then b := false
      end;
  done; !b

(*
let bss5 = [ bs | bs <- bss4; condition4 bs aa ]
*)

let apply phi (s,t) = Term.subst phi s, Term.subst phi t

let combine phi2 phi1 = 
  [ v, (Term.subst phi2) t | v,t <- phi1 ]
@ [ v,t | v,t <- phi2; not (List.mem v (List.map fst phi1)) ]

let lookup i basis_var =
 [ List.nth b i, var | b,var <- basis_var; List.nth b i >= 1 ]

let rec make_term f acc = function
  | [] -> F(f,acc)
  | (n,s) :: tt ->
      let ls = [ s | i <- Ls.range 1 n ] in
      make_term f (acc @ ls) tt
      
let make_term_or_sing f ls = 
  match ls with
  | [(n,t)] when n = 1 -> t
  | _ -> make_term f [] ls
  
let make_eqs f basis_var aa = 
 let n = (List.length aa) - 1 in 
 [ List.nth aa i, make_term_or_sing f n_var_ls | i <- Ls.range 0 n; 
   n_var_ls <- [lookup i basis_var]; not (is_var (List.nth aa i)) ]  

let s_of_v = function
  | V x -> x
  | _ -> failwith "invalid arg in Ac.s_of_v"

let make_sigma f basis_var aa =
 let n = (List.length aa) - 1 in 
 [ s_of_v (List.nth aa i), make_term_or_sing f n_var_ls | i <- Ls.range 0 n; 
   n_var_ls <- [lookup i basis_var]; (is_var (List.nth aa i)) ]  
  

let rec unify_set f i k sigma eqs =
  if i > k then sigma else
  let si,ti = List.nth eqs i in 
  let next_sigma = [ combine phi1 phi2 | phi2 <- sigma; 
		     phi1 <- f (Term.subst phi2 si,Term.subst phi2 ti) ] in
  unify_set f (i+1) k next_sigma eqs


let rec simplify_vt subst =
  let appears v sub1 =
    try 
      let _ = List.find (fun (x,y) -> List.mem v (Term.vars y)) sub1 in true 
    with Not_found -> false
  in
  let v,t = List.find (fun (x,y) -> appears x (Ls.del [x,y] subst)) subst in 
  let subst1 = Ls.del [v,t] subst in 
  [ w, Term.subst [v,t] s | w,s <- subst1 ]


let rec simplify subst = 
 try 
  let subst1 = simplify_vt subst in
  simplify subst1
 with Not_found -> subst

let remove_vars vars subst =
 [ v,t | v,t <- subst; List.mem v vars ]

let filter_invalid subs =
 [ sub | sub <- subs; List.for_all (fun (x,y) -> not (List.mem x (Term.vars y))) sub ]

let rec powerset = function
 | [] -> [[]]
 | h::t -> List.fold_left (fun xs t -> (h::t)::t::xs) [] (powerset t);;

(*
let rec unify ac = function
  | t1,t2 when t1 = t2 -> [[]]
  | (V x, t) 
  | (t, V x) when not (List.mem x (vars t)) -> [[x,t]]
  | F(f,ss),F(g,tt) when f=g && not (List.mem f ac) ->
      let k = (List.length ss) - 1 in
      let ss_tt = List.combine ss tt in 
      unify_set (unify ac) 0 k [[]] ss_tt 
  | F(f,ss),F(g,tt) when f=g && (List.mem f ac) ->
      let f,aa,bs = basic_sol [f] (F(f,ss)) (F(g,tt)) in
      let bs1 = filterPreCond4 bs aa in  
      let bss2 = [ b | b <- powerset bs1; b <> [] ] in
      let bss3 = [ b | b <- bss2; condition3 b aa ] in
      let bss4 = [ b | b  <- bss3; condition4 b aa ] in  
      let basis_var_ls = [ [ bj, Term.new_var () | bj <- bsi ] | bsi <- bss4 ] in 
      let subs =
      List.flatten
      [ unify_set (unify ac) 0 ((List.length eqs)-1)  [(make_sigma f basis_var aa)] eqs | basis_var <- 
	basis_var_ls; eqs <- [ make_eqs f basis_var aa ] ] 
      in subs (*
      let subs1 = List.map simplify subs in 
      List.map (remove_vars (Rule.variables (F(f,ss),F(g,tt)))) subs1 *)
  | _ -> []
*)
	

(* reconstruct term structure for given AC-term
   reconstruct f(x,y,z) => f(x,f(y,z)) *)
let rec reconstruct ac = function
    V x -> V x
  | F (f,ts) when not (List.mem f ac) -> F (f,ts)
  | F (f,ts) ->
     let c xs = F (f, xs) in
     match ts with
       [t1;t2] -> c [t1;t2]
     | t1::t2::ts' -> c [t1; reconstruct ac (c (t2::ts'))]
     | _ -> invalid_arg "reconstruct"

let rec unify_proto ac (s,t) =
  let s',t' = flatten ac s, flatten ac t in
  let vs = vars s @ vars t in
  let refresh sub = [x,v | x,v <- sub; List.mem x vs] in  (*
  unify_proto_aux ac vs (s',t')
   *)
  Ls.uniq
    [[x, reconstruct ac v | x,v <- sub]
    | sub0 <- unify_proto_aux ac vs (s',t'); sub <- [refresh sub0]]
and unify_proto_aux ac vs st = 
  let this = unify_proto_aux ac vs in
  match st with
  | t1,t2 when t1 = t2 -> [[]]
  | (V x, V y) -> if List.mem x vs then [[y,V x]]
                  else if List.mem y vs then [[x,V y]]
                  else [[x,V y]]
  | (V x, t) when not (List.mem x (vars t)) -> [[x,t]]
  | (t, V x) when not (List.mem x (vars t)) -> [[x,t]]
  | F(f,ss),F(g,tt) when f=g && not (List.mem f ac) ->
     let k = (List.length ss) - 1 in
     let ss_tt = List.combine ss tt in 
     unify_set this 0 k [[]] ss_tt
  | F(f,ss),F(g,tt) when f=g && (List.mem f ac) ->
     let vs,vt = [new_var() | _ <- ss], [new_var() | _ <- tt] in
     let e_sigmas = elementary_ac_unify ac (F(f,vs)) (F(f,vt)) in
     let list = [ List.combine (List.map (subst e_sigma) vs) ss @
                    List.combine (List.map (subst e_sigma) vt) tt
              | e_sigma <- e_sigmas ] in
     List.flatten
       [ unify_set this 0 (List.length ts -1) e_sigmas ts
       | ts <- list]
  | _ -> []

let is_psubterm s t = s <> t && List.mem s (subterms t)

let superset s eqs = [v | u,v <- eqs; is_psubterm s u]
                    @ [v | v,u <- eqs; is_psubterm s u]

let vars_in_equations eqs = [v | s,t <- eqs; v <- vars s @ vars t]
    
let build_sigma ac eqs =
  match [x, reconstruct ac t | V x,t <- eqs ]
        @ [x, reconstruct ac t | t,V x <- eqs] with
    [] -> []
  | sub -> sub

(* helper functions *)
let not_mem x xs = not (List.mem x xs)
let elem x eqs = List.mem x (vars_in_equations eqs)
let free x eqs = not_mem x (vars_in_equations eqs)

(* Variable Replacement *)
let rec var_rep org_vars eqs eq = match eq with
  | V x, V y when
         (not_mem x org_vars || List.mem y org_vars)
         && elem x eqs && elem y eqs
    -> Some (eq :: List.map (apply [x,V y]) eqs)
  (*
  | V y, V x when
         (not_mem x org_vars || List.mem y org_vars)
         && elem x eqs && elem y eqs
    -> Some (eq :: List.map (apply [x,V y]) eqs)
   *)
  | _ -> None

(* let _ = var_rep ["x"; "y"] [f[x;z],f[y]; g[V "c"],a; V "c", V "x";] (z,x)  *)
(* let _ = var_rep ["x"; "y"] [x,z; f[x;z],f[y]; g[V "c"],a; ] (V "c", V "x") *)

(* Replacement *)
let rec rep eqs eq = match eq with
  | (V x, V y) when x <> y && elem x eqs && elem y eqs
    -> Some (eq :: List.map (apply [x,V y]) eqs)
  | (V x, t) when elem x eqs && not_mem x (vars t)
    -> Some (eq :: List.map (apply [x,t]) eqs)
  (*
  | (t, V x) when elem x eqs && not_mem x (vars t)
    -> Some (eq :: List.map (apply [x,t]) eqs)
   *)
  | _ -> None

(* let _ = rep [f[x],f[y]; g[c],g[y]; x, z] (y,x)  *)
(* let _ = rep [f[x],f[y]; g[c],g[y]; x, z] (x,a)  *)
(* let _ = rep [f[x],f[y]; g[c],g[y]; x, z] (a,x) *)
(* let _ = rep [f[x],f[y]; g[c],g[y];] (g[x],x)   *)

(* Existential Quntifiers Elimination *)
let rec eqe org_vars eqs = function
  | (V x, t) when not_mem x org_vars && free x ((t,t)::eqs) ->
     Some eqs
  | (t, V x) when not_mem x org_vars && free x ((t,t)::eqs) ->
     Some eqs
  | _ -> None

(* let _ = eqe [] [y,x; f[x],f[y];] (z,x) *)

(* Merge *)
(* given eqs shoud be sorted by sort_eq *)
let rec merge eqs = function
  | (V x, V y)
    -> None
  | (V x, t) when List.exists
                    (fun (u,v) -> (u = V x && not (is_var v)))
                    eqs
    -> Some ((V x, t) :: [if u = V x then t,v else u,v | u,v <- eqs])
  | (t, V x) when List.exists
                    (fun (u,v) -> (u = V x && not (is_var v)))
                    eqs
    -> Some ((V x, t) :: [if u = V x then t,v else u,v | u,v <- eqs])
  | _ -> None

(* let _ = merge [f[x],f[y]; y, a; b,y] (y,g[x]) *)
(* let _ = merge [f[x],f[y]; y, a; y,b] (g[x],y) *)

(* Mutate *)
let rec mutate ac = function
  (* for standard unification *)
  | (F(f,ss), F(g,tt)) when f = g && not_mem f ac ->
     if List.length ss <> List.length tt
     then None
     else Some [ List.combine ss tt ]
  (* for AC-unification *)
  | (F(f,ss0), F(g,tt0)) when f = g && List.mem f ac ->
     let ss,tt = remove_common_args ss0 tt0 in
     let vs,vt = [new_var() | _ <- ss], [new_var() | _ <- tt] in
     let unifiers = elementary_ac_unify ac (F(f,vs)) (F(f,vt)) in
     Some [ List.combine [subst sigma v | v<-vs] ss @
              List.combine [subst sigma v | v<-vt ] tt
          | sigma <- unifiers ]
  | _ -> None

let rec mutate_all ac eqs =
  let eqss = List.map (mutate ac) eqs in
  if List.for_all ((=)None) eqss then 
    [eqs]
  else
    let singleton xs = List.map (fun x -> [[x]]) xs in
    let aux = function
        Some xs, _ -> xs
      | None, xs   -> xs in
    [eqs' | xss <- Ls.product (List.map aux
                                  (List.combine eqss (singleton eqs)));
            eqs' <- mutate_all ac (List.concat xss)]

(* let _ = mutate ["+"] (f[x;y], f[a;b])                      *)
(* let _ = mutate ["+"] (f[x], f[a;b])                        *)
(* let _ = mutate ["+"] (f[b;a], f[a;b])                      *)
(* let _ = mutate ["+"] (x +: y, a +: b)                      *)
(* let _ = mutate ["+"] (b +: a, a +: b)                      *)
(* let _ = mutate ["+"] (F ("+", [b; a;x]), a +: y)           *)
(* let _ = mutate_all ["+"] [x +: y, a +: b]                  *)
(* let _ = mutate_all ["+"] [x +: y, a +: b; g[x; y], g[a;b]] *)
(* let _ = mutate_all ["+"] [x +: y, a +: b; x +: y, a +: b]  *)

(* Check *)
(* False if there is no solution of eqs *)
let rec check eqs =
  if eqs = [] then true
  else
    let rec sig_check = function
        [] -> true
      | (F (f,_), F (g,_)) :: _ when f <> g -> false
      | eq :: eqs -> sig_check eqs in
    (* maybe [x, t | t,V x, <- eqs] is needed *)
    let sigma = [x, t | V x, t <- eqs] in
    let apply_right sigma (s,t) = s, subst sigma t in
    let eqs' = List.fold_left
                (fun eqs' s -> List.map (apply_right [s]) eqs')
                eqs
                sigma in
    sig_check eqs &&
    [] = [() | V x,F(f,ts) <- eqs'; List.mem x (vars (F (f,ts)))]

(* let _ = check [x,f[x]]                      *)
(* let _ = check [x,g[y]; a,b; y,g[z]; z,g[x]] *)
(* let _ = check [f[x],g[y]; a,b; x,g[y]]      *)
(* let _ = check [x,y; x,b]                    *)
(* let _ = check [x,y; a,b]                    *)
  
(* Prune *)
let prune org_vars eqs =
  let p x t = not_mem x org_vars && not (linear t) in
  [] = [t | V x,t <- eqs; p x t]

(* let _ = prune ["x";"y"] [z, f[x]]   *)
(* let _ = prune ["x";"y"] [z, x +: x] *)


let sort_eq (s,t) = match (s,t) with
    V _, V _ -> min s t, max s t
  | V _, F _ -> s, t
  | F _, V _ -> t, s
  | F _, F _ -> min s t, max s t
let cleanup eqs =
  Ls.uniq
    (List.filter (fun (s,t) -> s <> t)
       (List.map sort_eq eqs))

let split_by p xs = [x | x <- xs; p x], [x | x <- xs; not (p x)]

let occur_twice x eqs = 1 < List.length [s,t | s,t <- eqs; s = x || t = x]
                      
let condition1 org_vars eqs =
  let mem x = List.mem x org_vars in
  let p (s,t) = match s,t with
      V x, V y when mem x && mem y -> occur_twice s eqs || occur_twice t eqs
    | V x, _   when mem x          -> occur_twice s eqs
    | _  , V y when mem y          -> occur_twice t eqs
    | _        -> false in
  split_by p eqs

(* let _ = condition1 ["x"] [x,y;x,z;z,b] *)

(* "Syntactic" AC-Unification, A.Boudet and E.Contejean, 1994 *)
let unify_syn_aux_1 ac org_vars eqs =
  let rec loop used eqs0 =
    let eqs = cleanup eqs0 in
    let excepts, targets = condition1 org_vars eqs in
    let used' = used @ excepts in
    (* Format.printf "--eqs--@."; Trs.print_rules eqs;         *)
    (* Format.printf "--excepts--@."; Trs.print_rules excepts; *)
    match targets with
      [] -> [used@eqs]
    | eq :: eqs' -> 
       (* Format.printf "--targets--@."; Trs.print_rules [eq]; Trs.print_rules eqs'; *)
       match mutate ac eq with
         Some eqss ->
          (* Format.printf "--mutate--@."; List.iter Trs.print_rules eqss; *)
          [eqs | eqs1 <- eqss; eqs <- loop [] (eqs1@used'@eqs')]
       | None ->
       match merge eqs' eq with
         Some eqs1 ->
          (* Format.printf "merge: Some eqs1@."; Trs.print_rules eqs1; Format.printf "@."; *)
           loop [] (used'@eqs1)
       | None ->
       match var_rep org_vars (Ls.del (used@eqs) [eq]) eq with
         Some eqs1 ->
          (* Format.printf "var_rep: Some eqs1@."; Trs.print_rules eqs1; Format.printf "@."; *)
          loop [] eqs1
       | None ->
       match rep (Ls.del (used@eqs) [eq]) eq with
         Some eqs1 ->
          (* Format.printf "rep: Some eqs1@."; Trs.print_rules eqs1; Format.printf "@."; *)
          loop [] eqs1
       | None ->
       match eqe org_vars (used'@eqs') eq with
         Some eqs1 ->
          (* Format.printf "eqe: Some eqs1@."; Trs.print_rules eqs1; Format.printf "@."; *)
          loop [] (used'@eqs1)
       | None -> loop (eq::used') eqs'
  in
  loop [] eqs

(* let org_vars = ["x";"y";"z"] *)
(* let _ = unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[x;y], f[a;b]; g[x;x],g[y;b]] *)
(* let _ = unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[x;y], f[a;b]]                *)
(* let _ = unify_syn_aux_1 ["f"] ["x";"y";"z"] [a,b]                           *)

let unify_syn_aux_2 org_vars eqs =
  let rec loop used eqs0 =
    let eqs = cleanup eqs0 in
    (* Format.printf "--eqs--@."; Trs.print_rules eqs;         *)
    (* Format.printf "--excepts--@."; Trs.print_rules excepts; *)
    match eqs with
      [] -> [used]
    | eq :: eqs' -> 
       match merge eqs' eq with
         Some eqs1 ->
           loop [] (used@eqs1)
       | None ->
       match var_rep org_vars (Ls.del (used@eqs) [eq]) eq with
         Some eqs1 ->
          loop [] eqs1
       | None ->
       (* this is an orignal modification; not in the paper *)
       match eqe org_vars (used@eqs') eq with
         Some eqs1 ->
          loop [] (used@eqs1)
       | None -> loop (eq::used) eqs'
  in loop [] eqs

(* let _ = List.concat @@ List.map (unify_syn_aux_2 ["x";"y";"z"])                *)
(*         @@ List.filter check                                                   *)
(*         @@ unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[x;y], f[a;b]; g[x;x],g[y;b]] *)
(* let _ = List.concat @@ List.map (unify_syn_aux_2 ["x";"y";"z"])                *)
(*         @@ List.filter check                                                   *)
(*         @@ unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[x;y], f[a;b]]                *)

let rec solved org_vars eqs =
  let p = function
    | (V x,_) -> List.mem x org_vars
    | (_,V x) -> List.mem x org_vars
    | _ -> false
  in List.for_all p eqs

let unify_syn_aux_34 org_vars eqs =
  if not (prune org_vars eqs && check eqs) then
    []
  else if solved org_vars eqs then
    [eqs]
  else
    let rec loop used eqs0 =
      let eqs = cleanup eqs0 in
      match eqs with
        [] -> [used]
      | (V x,t) :: eqs' when not_mem x org_vars -> 
         begin
           match merge eqs' (V x,t) with
             Some eqs1 -> loop [] (used@eqs1)
           | None -> loop [] ((V x, t)::used)
         end
      | (t, V x) :: eqs' when not_mem x org_vars -> 
         begin
           match merge eqs' (t, V x) with
             Some eqs1 -> loop [] (used@eqs1)
           | None -> loop [] ((t, V x)::used)
         end
      | eq :: eqs' ->
         match rep (Ls.del (used@eqs) [eq]) eq with
           Some eqs1 -> loop [] eqs1
         | None -> loop (eq::used) eqs'
    in loop [] eqs

(* let _ = Ls.concat_map (unify_syn_aux_34 ["x";"y";"z"])                         *)
(*         @@ Ls.concat_map (unify_syn_aux_2 ["x";"y";"z"]) @@ List.filter check  *)
(*         @@ unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[x;y], f[a;b]; g[x;x],g[y;b]] *)
(* let _ = Ls.concat_map (unify_syn_aux_34 ["x";"y";"z"])                         *)
(*         @@ Ls.concat_map (unify_syn_aux_2 ["x";"y";"z"]) @@ List.filter check  *)
(*         @@ unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[x;y], f[a;b]]                *)
(* let _ = Ls.concat_map (unify_syn_aux_34 ["x";"y";"z"])                         *)
(*         @@ Ls.concat_map (unify_syn_aux_2 ["x";"y";"z"]) @@ List.filter check  *)
(*         @@ unify_syn_aux_1 ["f"] ["x";"y";"z"] [f[b;a], f[a;b]]                *)

let rec unify_syn ac eqs0 =
  let vs = Ls.uniq (vars_in_equations eqs0) in
  let eqs = [flatten ac s, flatten ac t | s,t <- eqs0] in
  Ls.uniq @@ List.map (build_sigma ac) @@ unify_syn_aux ac vs eqs
and unify_syn_aux ac org_vars eqs =
  Ls.concat_map (unify_syn_aux_34 org_vars)
  @@ Ls.concat_map (unify_syn_aux_2 org_vars)
  @@ unify_syn_aux_1 ac org_vars eqs
  
(* let _ = unify_syn ["f"] [f[x;y], f[f[a;a;b]]; g[x;x],g[y;b]] *)
(* let _ = unify_syn ["f"] [f[x;y], f[a;b;b]]                   *)
(* let _ = unify_syn ["f"] [f[b;a], f[a;b]]                     *)

let replace_var_by_consts t = 
 let vars = Term.vars t in
 let sigma = [v, Term.new_const () | v <- vars] in
  sigma, Term.subst sigma t

let rec replace_consts_by_var sigma = function
  | V x as t -> t
  | F(f,[]) as t ->
      begin
	try
	  let v,_ = List.find (fun (w,u) -> u = t) sigma in V v
	with Not_found -> t
      end
  | F(f,ss) ->  F(f,[ replace_consts_by_var sigma si | si <- ss ])

let unify ac (s,t) =
  unify_syn ac [s,t]

let matcher ac (p,t0) = 
  let sigma,t = replace_var_by_consts t0 in
  [[x, replace_consts_by_var sigma t | x,t <- phi]
  | phi <- unify ac (p,t)]

let unifiable ac (t1,t2) =
  [] <> unify ac (t1,t2)

