%{
open Num
open Lexing
open Term
open Parsing

let add xs rs (xs', rs') = (xs @ xs', rs @ rs')

let rec convert_term xs = function
  | V _ as t -> t
  | F (x, []) when List.mem x xs -> V x
  | F (f, ts) -> F (f, List.map (convert_term xs) ts)

let convert_rule xs (l, r) = (convert_term xs l, convert_term xs r)

let convert_rules (xs, rs) = List.map (convert_rule xs) rs

(* generate rules corresponding theory *)
let x,y,z = V "x", V "y", V "z"
let vs    = ["x";"y";"z"]

let func = fun x ts -> F (x,ts)

let formA1 f = let f = func f in
  f [f [x;y]; z], f [x; f[y;z]]

let formA2 f = let f = func f in
  f [x; f[y;z]], f [f [x;y]; z]

let formC f = let f = func f in
  f[x;y], f [y;x]

let rule_C  fs = [ formC f | f <- fs]
let rule_A  fs = [ rl | f <- fs; rl <- [formA1 f; formA2 f]]
let rule_AC fs = [ rl | f <- fs; rl <- [formA1 f; formC f]]

let syntax_error msg =
  let p = symbol_start_pos () in
  Format.eprintf "File %S at line %d, character %d:@.%s@." 
    p.pos_fname p.pos_lnum (p.pos_cnum - p.pos_bol + 1) msg;
  exit 1

let type_check = function
  | "COMMUTATION" -> ()
  | s             -> syntax_error ("Incorrect PROBLEM: " ^ s)

let choice thr fs = match thr with
  | "A"  -> rule_A  fs
  | "C"  -> rule_C  fs
  | "AC" -> rule_AC fs
  | thr  -> syntax_error ("Unexpected Theory " ^ thr)

%}

%token <string> IDENT
%token LPAREN RPAREN 
%token ARROW ARROWEQ COMMA EOF 
%token PROBLEM VAR RULES COMMENT THEORY OTHER STRATEGY
%token INNERMOST

%type <Term.trs list> toplevel
%type <Term.trs> system
%start toplevel system

%%

toplevel:
  | LPAREN PROBLEM IDENT RPAREN systems EOF
      { type_check $3; $5 }

systems:
  | comment_decl systems { $2 }
  | system systems       { $1 :: $2 }
  | any_decl systems     { $2 }
  |                      { [] }
  | error { syntax_error "Syntax error." }

system:
  | var_decl rules_decl { convert_rules ($1,$2) }
  | error               { syntax_error "Syntax error." }

var_decl:
  | LPAREN VAR vars RPAREN { $3 }
  |                        { [] }
  | error { syntax_error "Syntax error." }

rules_decl:
  | LPAREN RULES rules RPAREN { $3 }
  | error { syntax_error "Syntax error." }

comment_decl:
  | LPAREN COMMENT anylist RPAREN { () }
  |                               { () }
  | error { syntax_error "Syntax error." }

any_decl:
  | LPAREN IDENT anylist RPAREN any_decl { () }
  |                                      { () }
  | error { syntax_error "Syntax error." }

anylist:
  | any anylist { () }
  |             { () }

any:
  | LPAREN anylist RPAREN { () }
  | IDENT { () } 
  | ARROW { () }
  | COMMA { () }
  | OTHER { () }

vars:
  | IDENT vars { $1 :: $2 }
  |            { [] }

rules:
  | rule rules { $1 :: $2 }
  | { [] }

rule:
  | term ARROW term { ($1, $3) }
  | error      { syntax_error "Syntax error." }

term:
  | IDENT LPAREN terms RPAREN { F ($1, $3) }
  | IDENT                     { F ($1, []) }
  | error { syntax_error "Syntax error." }

terms:
  | term COMMA terms { $1 :: $3 }
  | term             { [$1] }
  |                  { [] }

theories:
  | LPAREN theory RPAREN theories { $2 @ $4 }
  | { [] }

theory:
  | IDENT syms { choice $1 $2 }

syms:
  | IDENT syms { $1 :: $2 }
  | { [] }
