(* logical formula to yices formula converter *)
open Form
open Format

(* util functions *)
let string_bin f x y = sprintf "(%s %s %s)" f x y
let string_uni f x   = sprintf "(%s %s)" f x

(* flatten by white space *)
let rec closure g fs = String.concat " " [ g f | f <- fs ]

(* convert to yices format *)
let rec term_to_string = function
  | A s       -> s
  | V (v,t)   -> v
  | BV bs     -> "0b" ^ String.concat "" [ if b then "1" else "0" | b <- bs ]
  | Tuple fs  -> string_uni "mk-tuple" (closure form_to_string1 fs)
and form_to_string1 = function
  | True         -> "true"
  | False        -> "false"
  | Term t       -> term_to_string t
  | N f          -> string_uni "not" (form_to_string1 f)
  | C []         -> "true"
  | C cnf        -> string_uni "and" (closure form_to_string1 cnf)
  | D []         -> "false"
  | D dnf        -> string_uni "or"  (closure form_to_string1 dnf)
  | OP (bin,fs)  -> string_uni bin   (closure form_to_string1 fs)
  | IMP (a,b)    -> string_bin "=>"  (form_to_string1 a) (form_to_string1 b)
  | EQ (a,b)     -> string_bin "="   (form_to_string1 a) (form_to_string1 b)
  | GT (a,b)     -> string_bin ">"   (form_to_string1 a) (form_to_string1 b)
  | LT (a,b)     -> string_bin "<"   (form_to_string1 a) (form_to_string1 b)

let form_to_string f = form_to_string1 (Form.simply f)

(* generate definition string *)
let type_to_string = function
  | BOOL        -> "bool"
  | INT         -> "int"
  | NAT         -> "nat"
  | REAL        -> "real"
  | BITVECTOR n -> sprintf "(bitvector %d)" n
let definition_to_string = function
  (* | V (v,NAT) -> sprintf "(define %s :: %s)(assert (>= %s 0))" v "real" v *)
  | V (v,t)   -> sprintf "(define %s :: %s)" v (type_to_string t)
  | _         -> invalid_arg "Yices.definitioon_to_string: bad argument"

(****************** GENERATOR *****************)

(** generate typed variable **)
let def_nat version xs =
  Ls.uniq
    begin
    if version = 1 then
      [ Term (V (x,NAT)) | x <- xs ]
    else
      [ Term (V (x,INT)) | x <- xs ] @
      [ GT (Term (V (x,INT)), Term (A "-1")) | x <- xs ]
    end

let def_bool css = Ls.uniq
  [ Term (V (cs, BOOL)) | cs <- css ]

let def_int css = Ls.uniq
  [ Term (V (cs, INT)) | cs <- css ]

(* definition generator *)
let gen_definition fs =
  [ definition_to_string v | v <- Ls.uniq [ v | f <- fs; v <- variables f ]]

(** assertion **)
let gen_assert fs = Ls.uniq
  [string_uni "assert" (form_to_string f) | f <- fs ]

(* max sat *)
let gen_assert_plus mx_fs =
  let max = [ n,f | MAX (n,f) <- mx_fs ]
  and nmx = [ f   | NMX f <- mx_fs ] in
  [ string_bin "assert+" (form_to_string f) (string_of_int n) | n,f <- max ]
  @ gen_assert nmx

(** check and modeling command **)
let gen_instruction = function
  | 1 -> [ "(check)" ]
  | 2 -> [ "(check)";"(show-model)" ]
  | n -> failwith (Format.sprintf "%d is unsupported Yices version" n)


(****************** TRANSLATE TOOL *****************)
(* all in one *)
let to_string version fs =
  gen_definition fs @
  gen_assert fs @
  gen_instruction version

(* yices2 hasn't max-sat function (3/10/2014) *)
let to_string_max mx_fs =
  let opt   = ["(set-evidence! true)"]
  and defs  = gen_definition (List.map max_to_form mx_fs)
  and forms = gen_assert_plus mx_fs
  and instruction = ["(max-sat)"] in
  opt @ defs @ forms @ instruction

(* abbreviations *)
let yices1 fs = to_string 1 fs
let yices2 fs = to_string 2 fs
let yices fs  = yices1 fs

let max_yices mx_fs = to_string_max mx_fs

(****************** SAT TOOL *****************)

(* formate cs to [("var",["val"])] *)
let align fs cs =
  let di = [ v, data | [v],data <- cs ] in
  let p x xs =
    try
      List.assoc x xs; true
    with _ -> false in
            (* [ v, data | data,[v] <- cs ] in *)
   (* [ v, List.assoc v di | f <- fs; V (v,_) <- variables f ] *)
     [ v, List.assoc v di | f <- fs; V (v,_) <- variables f; p v di ]

let sat fs = match Console.sat (yices fs) with
  | Some evd -> Some (Ls.uniq (align fs evd))
  | None     -> None 

(* let maxsat mx_fs = match Console.maxsat (max_yices mx_fs) with *)
let maxsat mx_fs = let f = Console.watcher "transformation" max_yices mx_fs in
  match Console.maxsat f with
  | Some (evd, cores, cost) ->
      Some (align (List.map max_to_form mx_fs) evd, cores, cost)
  | None -> None


(** for debug **)
let rec simple' = function
    | C ts  -> [ f | t <- ts; f <- simple' t ]
    | other -> [other]
let simple fs = [ f' | f <- fs; f' <- simple' f ]

let show fs        = List.iter print_endline @@ yices fs
let show_max mx_fs = List.iter print_endline @@ max_yices mx_fs
