open Ls
open Term
open Result

open Format
open Arg

let version = "1.5"

let usage = sprintf "\
Version: %s
Webpage: www.jaist.ac.jp/project/saigawa/

CoLL is commutation prover based on
Hindley's commutation theorem for left-linear TRS.

Usage: coll [options] COM_file [COM_file ...]
  1. prove commutation of a commutation problem P
    => coll [options] P
  2. prove commutation of TRSs a.trs and b.trs
    => coll [options] a.trs b.trs
  3. prove confluence of a TRS a.trs
    => coll [options] -cr a.trs"
    version

let read_from_stdin reader () =
  (* hiden experimental function *)
  let problem_str = Console.read_lines stdin in
  let temp_file = Filename.temp_file "inline_" ".COM" in
  Console.finally temp_file
    (fun f ->
      Console.write_file f problem_str;
      reader temp_file)
    Sys.remove

(*** this is Main module ***)let comm_checker r s =
  let rr,ss = Trs.sort r, Trs.sort s in
  if !Ref.no_comm  then
    if !Ref.tcap then
      match Commute.noncomm rr ss with
      | NO nf -> NO nf
      | other -> Commute.direct rr ss
    else
      Commute.direct rr ss
  else
    Commute.commute rr ss

let cr_checker r = comm_checker r r

(** for composable decomposition **)

let if_yes rss proofs =
  printf "YES\n@.";
  if !Ref.no_proof then
    ()
  else
    begin
      if !Ref.no_decompose then
        Trs.print_rules (union rss)
      else
        Compose.show_decompose rss;
        List.iter Commute.show_proof proofs
    end

let if_no rss (t,s,u) =
  printf "NO\n@.";
  if !Ref.no_proof then
    ()
  else
    begin
      if !Ref.no_decompose then
        Trs.print_rules (union rss)
      else
        Compose.show_decompose rss;
        printf "unjoinable peak@.";
        printf "%4s%s\n*<- %s ->*\n%4s%s@."
          "" (sprint t) (sprint s) "" (sprint u);
    end

let if_maybe rs =
  printf "MAYBE\n@."
  (*
  if_no_proof @@
    Compose.print_decomps rs;
    printf "commutation of following TRSs cannot be solved@.";
    printf "--- R\n%s\n--- S\n%s@."
    (Trs.sprint_rules r)
    (Trs.sprint_rules s)
   *)

(* addition of rules may generate non-left-linear rules *)
let preprocess rs =
  if !Ref.reduce      
    then Trans.reduce !Ref.l rs else
  if !Ref.fc          
    then Trans.filter (Trans.forwarding   !Ref.l rs) else
  if !Ref.reduce_joins
    then Trans.reduce_joins !Ref.l rs else
  if !Ref.add_joins
    then Trans.filter (Trans.add_joins    !Ref.l rs)
  else (true,rs)

let check_comm rr ss =
  if not !Ref.no_ll && not (Trs.left_linear (rr@ss)) then
    print_endline "MAYBE\ninput TRSs are not left-linear"
  else
    let (persist1,r),(persist2,s) = preprocess rr, preprocess ss in
    match comm_checker r s with
    | YES proof ->
       if_yes [r;s] [proof];
       Trans.is_transformed rr r; Trans.is_transformed ss s;
    | NO nfp when persist1 && persist2 ->
       if_no  [r;s] nfp;
       Trans.is_transformed rr r; Trans.is_transformed ss s;
    | _     -> if_maybe r

let check_cr rs =
  if not !Ref.no_ll && not (Trs.left_linear rs) then
    print_endline "MAYBE\ninput TRSs are not left-linear"
  else
    let persist,r = preprocess rs in
    let modular = if !Ref.no_decompose then [r] else Compose.decomposeCOM r in
      (* if !Ref.no_decompose then [r] else Compose.cons_share r *)
    match Result.sequence_with cr_checker modular with
    | YES proofs ->
       if_yes modular proofs; Trans.is_transformed rs r;
    | NO nfp when persist ->
       if_no  modular nfp   ; Trans.is_transformed rs r;
    | _ -> if_maybe r

(* ------------------------------------------------------ *)
(*
  System Architecture: suppose input files are
   - a COM_file -> prove commutation of TRSs
   - TRS, TRS   -> prove commutation of TRS 1 and TRS 2
   - TRS^n      -> prove commutation each other
   - TRS        -> prove CR of TRS if -cr is given
*)
(* ------------------------------------------------------ *)

(* check all dependent tools *)
let env_test () =
  let succ s  = printf "%s@." s
  and error s = printf "%s@." s; exit 1 in
  let trying f x s e =
    try
      begin ignore (f x); succ s end
    with _ -> error e in
  if not !Ref.test
  then
    ()
  else
    begin
      printf "Testing...@.";
      trying Smt2.sat_check () "SMT solver exists" "SMT solver does not exist";
      printf "Done@.";
      exit 0
    end

let recognize_systems trss =
  List.iter (Trs.label) trss

let run_comm progs =
  (* exit if unsupported flags are given *)
  Ref.commute_flag_check ();
  Ref.commute_flag_on ();
  (* verbose *)
  if not (!Ref.verbose) then ()
  else (eprintf "mode: cr@."; Ref.print_options (); Ref.print_flags ())
  ;
  (* parse TRS *)
  let rss = match progs with
    | []    when !Ref.use_stdin -> read_from_stdin Coll_read.read_problem ()
    | []    -> (eprintf "ERROR\nno inputs@."; exit 1)
    | [p]   -> Coll_read.read_problem p
    | progs -> List.map Coll_read.read_trs progs in
  List.iter
    (fun (r,s) -> recognize_systems [r;s]; check_comm r s)
    [ Trs.init r, Trs.init s | r,s <- Listx.tri rss ]

let run_cr progs =
  Ref.confluence_flag_on ();
  (* verbose *)
  if not (!Ref.verbose) then ()
  else (eprintf "mode: cr@."; Ref.print_options (); Ref.print_flags ())
  ;
  (* parse TRS and check it *)
  match progs with
    [] when !Ref.use_stdin ->
      let rs = read_from_stdin Coll_read.read_trs () in
      recognize_systems [rs];
      check_cr rs
  | [f] ->
      let rs = Coll_read.read_trs f in
      recognize_systems [rs];
      check_cr rs
  | _ -> (eprintf "ERROR\ntoo many inputs@."; exit 1)


(***************************************************************)
(*********************** OPTION SETTINGS ***********************)
let item s = "", Unit (fun () -> ()), "\n"^s

let functions = [
   item "Functions:";
   ("-k"            , Set_int Ref.k       , "<x>" ^ sprintf " use at most x steps for rewriting (default is %d)" !Ref.k);
   ("--confluence"  , Set Ref.cr_mode     , "" );
   ("-cr"           , Set Ref.cr_mode     , " prove confluence of input system (long is --confluence)" );
   ("--no-proof"    , Set Ref.no_proof    , "" );
   ("-np"           , Set Ref.no_proof    , " no proof (long is --no-proof)");
   ("--no-comm"     , Set Ref.no_comm     , "" );
   ("-nc"           , Set Ref.no_comm     , " disable commutative decomposition to apply only single theorem (long is --no-comm)");
   ("--no-decompose", Set Ref.no_decompose, "");
   ("-nd"           , Set Ref.no_decompose, " disable decomposition by modularity (long is --no-decompose)");
   ("--no-ll"       , Set Ref.no_ll       , "");
   ("-nll"          , Set Ref.no_ll       , " allow non-left-linear TRS (resulting proof can be wrong) (long is --no-ll)");
]

let transformations = [
   item "Transformations:";
   ("--forwarding"  , Set Ref.fc          , "");
   ("-fw"           , Set Ref.fc          , " enable forward closure (long is --forwarding)");
   ("--add-joins"   , Set Ref.add_joins   , "");
   ("-add-js"       , Set Ref.add_joins   , " enable redundant rule addition (long is --add-joins)");
   ("--reduce"      , Set Ref.reduce      , "");
   ("-red"          , Set Ref.reduce      , " enable redundant rule elimination w.r.t -> (long is --reduce)");
   ("--reduce-joins", Set Ref.reduce_joins, "");
   ("-red-js"       , Set Ref.reduce_joins, " enable redundant rule elimination w.r.t <-> (long is --reduce-joins)");
   ("-l"            , Set_int Ref.l       , "<x>" ^ sprintf " use at most x steps for fowarding/reducing (default is %d)" !Ref.l);
]

let thm_specs = [
  item "Theorem-Flags: if some blow flags given, other flags are disabled";
  ("-dc", Set Ref.dc, " use development closedness");
  ("-mo", Set Ref.mo, " use mutual orthogonality");
  ("-sc", Set Ref.sc, " use strongly commutation (Huet 80)");
  ("-rl", Set Ref.rl, " use rule labeling (V Oostrom 2008)");
  (* jk family *)
  ("-kb"  , Set Ref.kb  , " use Knuth and Bendix criterion");
  ("-jk"  , Set Ref.jk  , " use Jouannaud and Kirchner (1986) with E = {A,C,AC}");
  ("-jka" , Set Ref.jka , " same as -jk with E = {A}");
  ("-jkc" , Set Ref.jkc , " same as -jk with E = {C}");
  ("-jkac", Set Ref.jkac, " same as -jk with E = {AC}");
  (* TCAP *)
  ("-tcap", Set Ref.tcap, " use TCAP approximation (unjoinability checking).");
  (* closedness *)
  ("-pc"  , Set Ref.pc, " [temp] use parallel closedness (Huet 1980)");
  ("-toyama81", Set Ref.toyama81,     " use Toyama's extended parallel closedness (1981)");
  ("-almost-pc", Set Ref.almost_pc,   " [temp] use almost parallel closedness (Toyama 1988)");
  ("-gramlich96", Set Ref.gramlich96, " [temp] use Gramlich's extended parallel closedness (1996)");
  ("-sic" , Set Ref.sic , " use simultaneous closedness");
  ("-uc", Set Ref.uc,   " [temp] upside closedness (Ohta and Oyamaguchi 1997)");
  ("-oc", Set Ref.oc,   " [temp] outside closedness (Ohta and Oyamaguchi 2004)");
  (* ("-opc", Set Ref.opc, " [temp] use outside parallel closedness"); *)
]

let tool_specs = [
  item "External Tools:";
  ("--smt", String Ref.set_smt_solver, " use specified SMT2 solver (default is " ^ !Ref.smt_solver ^ ")")
]

let debug_specs = [
  item "Debug-Options:";
  ("--stdin"    , Set Ref.use_stdin   , " read strings as an input from stdin");
  ("--verbose"  , Set Ref.verbose     , " show computing progress");
  ("--test"     , Set Ref.test        , " test your computer environment");
  ("--show-time", Set Ref.show_time   , " show each processing time");
  ("--dump"     , String Ref.set_dump , " use specified path for SMT2 solver");
  ("--off"      , Clear Ref.flag      , "" (* hidden option *))
]


(* stack of input file names which are
   - 'a' commutation problem, or
   - TRSs
*)
let progs = ref ([] : string list)

(* main *)
let main () = parse
  (align @@
      functions
    @ transformations
    @ thm_specs
    @ tool_specs
    @ debug_specs
    @ [item "Others:"] (* vertical skip (split to Debug and help) *)
  )
  (* accumulate TRS *)
  (fun p -> progs := !progs @ [p])
  usage
  ; (* end of Arg.parse *)
  env_test ();
  Step.cache_on (); Join.cache_on (); Commute.cache_on ();
  (* check running mode flag *)
  if !Ref.cr_mode
  then run_cr !progs
  else run_comm !progs
