open Ls
open Format

(* ---------------------------------------- *)
(***** interface for command line tools *****)
(* ---------------------------------------- *)
type command 'a = (string -> 'a) -> string * out_channel -> string list -> 'a

(* measure executing time *)
let chronicle f x =
  let time = Unix.gettimeofday () in
  let fx = f x in
  fx, (Unix.gettimeofday () -. time)

let watcher description f x =
  let fx, time = chronicle f x in
  if !Ref.show_time then
    begin
      Format.eprintf "%s\t: %.5fs@." description time;
      fx
    end
  else
    fx

let watcher_l description f x =
  watcher (description^"\n\t") f x

let write_channel out_ch =
  watcher "writing\t"
    (List.iter (fun line ->
      fprintf (formatter_of_out_channel out_ch) "%s@." line))

let write_file file lines =
  write_channel (open_out file) lines

(* IO functions *)
(* ---------------------------------------- *)
(* g is closer *)
let finally x f g =
  try 
    let y = f x in g x; y
  with
    e -> g x; raise e

(* read string list from stdin *)
let rec read_lines in_ch =
  try
    let line = input_line in_ch in
    line :: read_lines in_ch
  with End_of_file ->
    close_in in_ch;
    []

(* join by '\n'; read string from stdin *)
let read_all in_ch = String.concat "\n" (read_lines in_ch)

let close_temp_file (tmp, ch) =
  close_out ch ;
  Sys.remove tmp

(* exe CMD on CLI -> read all stdout (string) *)
let run command : string =
  let run' () =
    finally
      (Unix.open_process_in command)
      (fun ch -> read_all ch)
      (fun ch -> ignore (Unix.close_process_in ch))
  in watcher_l ("running\t"^command) run' ()

(* difference of run is "string list" *)
let run_lines command : string list =
  finally
    (Unix.open_process_in command)
    (fun ch -> read_lines ch)
    (fun ch -> ignore (Unix.close_process_in ch))


(* run with file *)
let run_with_body
    (closing   : string * out_channel -> unit)
    (f_command : string -> 'a)
    (path      : string * out_channel)
    (lines     : string list)
    = let exec (f, out_ch) =
      write_channel out_ch lines; f_command f in
    finally path exec closing

(* run IO function with file -> read all stdout *)
let run_with f_command =
  run_with_body close_temp_file f_command

(* not remove file *)
let run_with_save f_command =
  run_with_body (fun (_,ch) -> close_out ch) f_command

