(* List operators, and Set operator *)

(****** for List ******)
(** Mapping **)
let rec inner s f = function
  | []      -> s
  | x :: xs -> inner (f x :: s) f xs

let map_tail f xs = inner [] f xs

let concat_map f xs = List.concat @@
  map_tail f xs

(** Taking and Dropping **)
let rec take n xs = match n,xs with
  | m, _ when m <= 0 -> []
  | _, []            -> []
  | _, y::ys         -> y :: take (n-1) ys

let rec take_while f = function
  | y :: ys when f y -> y :: take_while f ys
  | _                -> []

let rec drop n xs = match n,xs with
  | m, _ when m <= 0 -> xs
  | _, []            -> []
  | _, y::ys         -> drop (n-1) ys

let rec drop_while f = function
  | y :: ys when f y -> drop_while f ys
  | ys               -> ys

(** Uniqueness **)
(* uniq_by works well if eq is equivalent *)
let rec uniq_by' zs eq = function
  | []      -> List.rev zs
  | x :: xs -> uniq_by' (x::zs) eq
        [y | y <- xs; not (eq x y)]

let uniq_by eq xs = uniq_by' [] eq xs

let uniq xs : 'a list =
  try uniq_by (=) xs
  with Stack_overflow -> failwith "Ls.uniq: stack overflow"


(** Delete **)
let rec remove ?eq:(eq=(=)) x = function
  | []                  -> []
  | y :: ys when eq x y -> ys
  | y :: ys             -> y :: remove ~eq:(=) x ys

let remove_by eq = remove ~eq:eq

let rec del ?eq:(eq=(=)) xs ys = match xs,ys with
  | []   ,_  -> []
  | xs   ,[] -> xs
  | x::xs,ys -> if List.mem x ys
    then del ~eq:eq xs (remove ~eq:eq x ys)
    else x :: del xs ys

let rec del_by eq = del ~eq:eq

let (\\) xs ys = del xs ys

(** Difference **)
let rec diff xs ys = match (xs,ys) with
  | x::s1, y::s2 ->
      if x = y then diff s1 s2 else xs, ys
  | _ -> xs,ys


(** Ranges **)
let rec range x y =
  if x > y then
    []
  else
    x :: range (x+1) y

let range_of i xs = range i @@ List.length xs + i - 1

let indexing ?n:(n=1) xs =
  try
    List.combine (range_of n xs) xs
  with _ ->
    failwith "Ls.indexing: cannot combine lists of different lengths"

let rec count = function
  | [] -> []
  | x :: _ as l ->
      let xs, ys = List.partition (fun y -> x = y) l in
      (x, List.length xs) :: count ys

(** Repeat **)
let rep n x = [ x | _ <- range 1 n ]


(** Replacing **)
(* head index is 0 *)
let replace_at ?n:(n=0) i x xs =
  [ if i = j then x else y | j,y <- indexing ~n:n xs ]


(** Zipping **)
(* [1;2;3] -> [[1;2;3];[1;2];[1];[]] *)
let rec prefix = function
  | []      -> [[]]
  | x :: xs -> [] ::
    List.map (fun ys -> x::ys) (prefix xs)

(* [1;2;3] -> [[1;2;3];[2;3];[3];[]] *)
let rec suffix xs =
  match xs with
  | []      -> [[]]
  | _ :: ys -> xs :: suffix ys

(* [1;2;3] -> [(1,2);(1,3);(2,3)] *)
let triangle xs =
  [ x,y | x::ys <- suffix xs; y <- ys ]

(* [1;2;3] -> [[1;2];[1;3];[2;3]] *)
let diagonal xs = 
  [ [x;y] | x,y <- triangle xs ]


(****** for Set ******)
let setminus xs ys = [x | x <- xs; not (List.mem x ys)]
let (-\\) = setminus

(** Basic Operator **)
let cup a b =
  let s = uniq a in s @ (b -\\ s)
let (@+) = cup


let cap a b = [x | x <- a; List.mem x b]
let (@^) = cap

let union xss = List.fold_left cup [] xss

let intersect = function
  | []      -> []
  | xs::xss -> List.fold_left cap xs xss


(** Subset relation **)
let subseteq a b =
  List.for_all (fun x -> List.mem x b) a

let supseteq b a = subseteq a b

let seteq a b =
  subseteq a b && subseteq b a
let (===) = seteq

let subset a b = not (a === b) && subseteq a b
let supset a b = not (a === b) && supseteq a b

(** Element **)
let minimal geq xs = (* geq = >= *)
  uniq [x | x <- xs; List.for_all (fun y -> not (geq x y) || geq y x) xs]

let maximal geq xs = (* geq = >= *)
  uniq [x | x <- xs; List.for_all (geq x) xs]

let choice ord a b =
  if ord a b then a else b

let minimum geq = function
  | []      -> failwith "no elements in list"
  | x :: xs -> List.fold_left (choice (fun x y -> not (geq x y))) x xs

let maximum geq = function
  | []      -> failwith "no elements in list"
  | x :: xs -> List.fold_left (choice geq) x xs


(** Set to Set of Set **)
let rec pset ps = function
  | [] -> ps
  | x :: xs ->
      pset
        (concat_map (fun ys -> [x::ys;ys]) ps)
        xs
(* if no rev then the powset break order of list s *)
let powset s = pset [[]] (List.rev s)

(* [[1];[2;3]] -> [[1;2];[1;3]] *)
let rec product = function
  | []        -> [[]]
  | xs :: xss -> [x :: ys | x <- xs; ys <- product xss]


(** Grouping **)
(* disjoint split *)
(* [ys,xs \\ ys | ys <- powset xs] *)
let rec dj_split = function
  | []      -> [[],[]]
  | x :: xs ->
  let dj = dj_split xs in
  [(x::ys,zs) | (ys,zs) <- dj] @
  [(ys,x::zs) | (ys,zs) <- dj]

(* quotient set for predicate p *)
let eq_class a p xs = uniq [x | x <- xs; p a x]
let rec quotset p xs =
  uniq_by seteq [eq_class x p xs | x <- xs]

(* for not equivalent relation *)
let rec group_by p xs =
  merge (quotset p xs)
and merge = function
  | []      -> []
  | xs::xss ->
    let up =
        List.filter
          (fun ys -> not (subset ys xs))
          xss in
  if List.exists (subset xs) up
    then
        merge up
    else
        xs :: merge up
