open Format
open Formatx

type t = Rule.t list

let print ppf rules =
  fprintf ppf "@[<v 0>%a@]" (print_list Rule.print "@ ") rules

let print_with sep ppf rules =
  fprintf ppf "@[<v 0>%a@]" (print_list (Rule.print_with sep)  "@ ") rules

let variables rules =
  Listx.unique [ x | rule <- rules; x <- Rule.variables rule ]

let functions rules =
  Listx.unique [ x | rule <- rules; x <- Rule.functions rule ]

let signature rules =
  Listx.unique [ x | rule <- rules; x <- Rule.signature rule ]

let defined_symbols rules =
  Listx.unique [ Term.root l | l, _ <- rules ]

let is_non_root_overlapping s t = not (Substitution.unifiable (Term.rename s) t)

let is_non_overlapping rules =
  List.for_all (fun (l1, s1, r1) ->
    List.for_all (fun (l2, r2) -> 
      ((l1, r1) = (l2, r2) && l1 = s1) ||
      is_non_root_overlapping s1 l2) rules)
    [ l1, s1, r1 | l1, r1 <- rules; s1 <- Term.subterms l1;
      not (Term.is_variable s1) ]

let is_non_duplicating rules = List.for_all Rule.is_non_duplicating rules

let variable_condition rules =
  List.for_all Rule.variable_condition rules

let is_constructor_system rules =
  let ds = defined_symbols rules in
  let p = function
    | Term.V _, _ -> true
    | Term.F (_, ts), _ -> 
        List.for_all (fun t -> Listset.inter (Term.functions t) ds = []) ts in
  List.for_all p rules

let wf rules = List.for_all Rule.wf rules

let linear rules = List.for_all Rule.linear rules

let left_linear rules = List.for_all Rule.left_linear rules

let right_linear rules = List.for_all Rule.right_linear rules

let rec remove rls = function
 | [] -> rls
 | st :: stt ->
     remove (Rule.remove rls st) stt

let flatten ac ls = 
 [ Term.flatten ac s, Term.flatten ac t | s,t <- ls ]
