open Unix
open Format
open Term
open Result

(* pretty printers *)

let print_list pr sep ppf = function
  | [] -> ()
  | x :: xs -> pr ppf x; List.iter (fprintf ppf (sep ^^ "%a") pr) xs 

let rec print_term ppf = function
  | V x -> fprintf ppf "%s" x
  | F (f, ts) -> fprintf ppf "%s(%a)" f (print_list print_term ",") ts

let print_rule sep ppf (l, r) =
  fprintf ppf "%a %s %a" print_term l sep print_term r

let print_trs ppf (strict_rules, weak_rules, ac_sym) =
  fprintf ppf "(VAR %a)@.(RULES@.%a@.%a@.)@."
    (print_list pp_print_string " ") 
    (Rules.variables (strict_rules @ weak_rules))
    (print_list (print_rule "->")  "@.") strict_rules
    (print_list (print_rule "->=") "@.") weak_rules;
  if ac_sym <> [] then
    fprintf ppf "(THEORY (AC %a))@."
      (print_list pp_print_string " ") ac_sym

(* call termination tool s*)

let try_finally x ~f g =
  try let y = f x in g x; y with e -> g x; raise e

let sn tool ?timeout ?(ac_sym = []) strict_rules weak_rules =
  if strict_rules = [] then YES () else begin
    let s = Filename.temp_file "" ".trs" in
    try_finally (open_out s) close_out ~f:begin fun ch -> 
      fprintf (formatter_of_out_channel ch) "%a%!" 
	print_trs (strict_rules, weak_rules, ac_sym)
    end;
    let result =
      try
	try_finally 
	  (open_process_in 
	     (match timeout with 
	      | None   -> sprintf "%s %s"      tool s
	      | Some f -> sprintf "%s %s %.2f" tool s f))
	  ~f:input_line
	  close_process_in
      with 
	End_of_file -> "MAYBE" in
    Sys.remove s;
    match result with
    | "YES" -> YES ()
    | "NO"  -> NO ()
    | _     -> MAYBE
  end
